/*
 * 8085 Cross Assembler
 *
 * ?COPY.TXT 1983-2005 Dave Dunfield
 * **See COPY.TXT**.
 */
#include <stdio.h>
#include <ctype.h>

#include "xasm.h"

/* 8085 opcode table - format: text, type, length, opcode */
static char opcodes[] = {
	'M','O','V'+128, 5, 1, 0x40,
	'M','V','I'+128, 6, 2, 0x06,
	'L','X','I'+128, 7, 3, 0x01,
	'I','N','R'+128, 3, 1, 0x04,
	'I','N','X'+128, 4, 1, 0x03,
	'D','C','R'+128, 3, 1, 0x05,
	'D','C','X'+128, 4, 1, 0x0b,
	'J','M','P'+128, 8, 3, 0xc3,
	'J','N','Z'+128, 8, 3, 0xc2,
	'J','Z'+128,     8, 3, 0xca,
	'J','C'+128,     8, 3, 0xda,
	'J','N','C'+128, 8, 3, 0xd2,
	'J','M'+128,     8, 3, 0xfa,
	'J','P'+128,     8, 3, 0xf2,
	'J','P','E'+128, 8, 3, 0xea,
	'J','P','O'+128, 8, 3, 0xe2,
	'C','A','L','L'+128, 8, 3, 0xcd,
	'C','N','Z'+128, 8, 3, 0xc4,
	'C','Z'+128,     8, 3, 0xcc,
	'C','C'+128,     8, 3, 0xdc,
	'C','N','C'+128, 8, 3, 0xd4,
	'C','M'+128,     8, 3, 0xfc,
	'C','P'+128,     8, 3, 0xf4,
	'C','P','E'+128, 8, 3, 0xec,
	'C','P','O'+128, 8, 3, 0xe4,
	'R','E','T'+128, 1, 1, 0xc9,
	'R','N','Z'+128, 1, 1, 0xc0,
	'R','Z'+128,     1, 1, 0xc8,
	'R','C'+128,     1, 1, 0xd8,
	'R','N','C'+128, 1, 1, 0xd0,
	'R','M'+128,     1, 1, 0xf8,
	'R','P'+128,     1, 1, 0xf0,
	'R','P','E'+128, 1, 1, 0xe8,
	'R','P','O'+128, 1, 1, 0xe0,
	'C','M','P'+128, 9, 1, 0xb8,
	'C','P','I'+128, 2, 2, 0xfe,
	'P','U','S','H'+128,4, 1, 0xc5,
	'P','O','P'+128, 4, 1, 0xc1,
	'A','D','D'+128, 9, 1, 0x80,
	'A','D','C'+128, 9, 1, 0x88,
	'S','U','B'+128, 9, 1, 0x90,
	'S','B','B'+128, 9, 1, 0x98,
	'A','D','I'+128, 2, 2, 0xc6,
	'A','C','I'+128, 2, 2, 0xce,
	'S','U','I'+128, 2, 2, 0xd6,
	'S','B','I'+128, 2, 2, 0xde,
	'A','N','A'+128, 9, 1, 0xa0,
	'A','N','I'+128, 2, 2, 0xe6,
	'X','R','A'+128, 9, 1, 0xa8,
	'X','R','I'+128, 2, 2, 0xee,
	'O','R','A'+128, 9, 1, 0xb0,
	'O','R','I'+128, 2, 2, 0xf6,
	'L','D','A','X'+128,4,1,0x0a,
	'L','H','L','D'+128,8,3,0x2a,
	'S','T','A','X'+128,4,1,0x02,
	'S','H','L','D'+128,8,3,0x22,
	'E','I'+128,1,1,0xfb,
	'D','I'+128,1,1,0xf3,
	'H','L','T'+128,1,1,0x76,
	'I','N'+128,2,2,0xdb,
	'O','U','T'+128, 2, 2, 0xd3,
	'P','C','H','L'+128,1,1,0xe9,
	'S','P','H','L'+128,1,1,0xf9,
	'L','D','A'+128, 8, 3, 0x3a,
	'S','T','A'+128, 8, 3, 0x32,
	'S','T','C'+128, 1, 1, 0x37,
	'C','M','C'+128, 1, 1, 0x3f,
	'C','M','A'+128, 1, 1, 0x2f,
	'X','C','H','G'+128,1,1,0xeb,
	'X','T','H','L'+128,1,1,0xe3,
	'R','A','L'+128, 1, 1, 0x17,
	'R','A','R'+128, 1, 1, 0x1f,
	'R','R','C'+128, 1, 1, 0x0f,
	'R','L','C'+128, 1, 1, 0x07,
	'D','A','D'+128, 4, 1, 0x09,
	'D','A','A'+128, 1, 1, 0x27,
	'N','O','P'+128, 1, 1, 0x00,
	'S','I','M'+128, 1, 1, 0x30,
	'R','I','M'+128, 1, 1, 0x20,
	'R','S','T'+128,10, 1, 0xc7,
/* Directives */
	'E','Q','U'+128, 100, 0, 0,
	'O','R','G'+128, 101, 0, 0,
	'D','B'+128,     102, 0, 0,
	'D','W'+128,     103, 0,-1,
	'D','R','W'+128, 103, 0, 0,
	'D','S'+128,     104, 0, 0,
	'S','T','R'+128, 105, 0, 0,
	'S','T','R','H'+128,105,0,1,
	'S','T','R','Z'+128,105,0,2,
	'F','C','B'+128, 102, 0, 0,
	'F','D','B'+128, 103, 0,-1,
	'R','D','B'+128, 103, 0, 0,
	'R','M','B'+128, 104, 0, 0,
	'F','C','C'+128, 105, 0, 0,
	'F','C','C','H'+128,105,0,1,
	'F','C','C','Z'+128,105,0,2,
	'E','N','D'+128, 106, 0, 0,
	'P','A','G','E'+128,120,0,0,
	'T','I','T','L','E'+128,121,0,0,
	'S','P','A','C','E'+128,122,0,0,
	'L','I','S','T'+128,123,0,0,
	'N','O','L','I','S','T'+128,124,0,0,
	'L','I','N','E'+128,125,0,0,
	0 } ;

/* error message text table */
static char *etext[] = { "?",
	"Unknown instruction",
	"Out of range",
	"Invalid addressing mode",
	"Invalid register specification",
	"Undefined symbol",
	"Invalid expression syntax",
	"Invalid argument format",
	"Improperly delimited string" };

/* Symbol table & free pointer */
char symtab[SYMB_POOL], *symtop;

/* misc. global variables */
char opcode, length, type, buffer[LINE_SIZE+1], label[SYMB_SIZE+1], operand[80],
	title[50], *optr;
unsigned char instruction[80], outrec[35];

unsigned addr, value, line, pcount, emsg, ecnt, lcount, ocnt,
	pagel=59, pagew=80;

char symf = 0, fulf = 0, intel = 0, quietf = 0, nlist = 0, casf = 0;

FILE *asm_fp, *hex_fp, *lst_fp;

/*
 * Define a symbol in the symbol table
 */
char *create_symbol(symbol, value)
	char *symbol;
	int value;
{
	register char *ptr, *ptr1;

	ptr = ptr1 = symtop;
	while(*symbol)
		*++ptr = *symbol++;
	*ptr1 = ptr - ptr1;
	*++ptr = value >> 8;
	*++ptr = value;
	if(ptr < (symtab+sizeof(symtab))) {
		symtop = ptr + 1;
		return ptr1; }
	return 0;
}

/*
 * Lookup a symbol in the symbol table
 */
char *lookup_symbol(symbol)
	char *symbol;
{
	register int l;
	register unsigned char *ptr, *ptr1;
	char *ptr2;

	ptr = symtab;
again:
	if((ptr2 = ptr) >= symtop) {
		value = 0x8888;
		return 0; }
	ptr1 = symbol;
	l = *ptr++ & SYMMASK;
	while(l--) {
		if(*ptr1++ != *ptr++) {
			ptr += l + 2;
			goto again; } }
	if(*ptr1) {
		ptr += 2;
		goto again; }
	value = *ptr++ << 8;
	value |= *ptr;
	return ptr2;
}

/*
 * Set the value for a symbol in the symbol table
 */
set_symbol(ptr, value)
	char *ptr;
	unsigned value;
{
	ptr += (*ptr & SYMMASK);
	*++ptr = value >> 8;
	*++ptr = value;
}

/*
 * Display the symbol table (sorted)
 */
dump_symbols()
{
	unsigned char *ptr, *hptr;
	unsigned int l, h, w;

	fprintf(lst_fp, "SYMBOL TABLE:\n\n");
	w = 0;
	for(;;) {
		for(hptr = symtab; hptr < symtop; hptr += (*hptr & SYMMASK) + 3)
			if(!(*hptr & SYMSORT))
				goto found;
		putc('\n', lst_fp);
		return;
found:	for(ptr = (*hptr & SYMMASK) + hptr + 3; ptr < symtop; ptr += (*ptr & SYMMASK) + 3) {
			if(*ptr & SYMSORT)
				continue;
			if(compsym(ptr, hptr) < 0)
				hptr = ptr; }
		*(ptr = hptr) |= SYMSORT;
		h = l = *ptr++ & SYMMASK;
		if((w + l + 7) >= pagew) {			/* Overrun page length */
			putc('\n', lst_fp);
			w = 0; }
		if(w) {								/* Not a start of line - separate */
			fprintf(lst_fp, "   ");
			w += 3; }
		while(l--)
			putc(*ptr++, lst_fp);
		w += (l = (h > 8) ? 24 : 8) + 5;	/* Calculate extended length */
		while(h++ < l)
			putc(' ', lst_fp);
		l = *ptr++ << 8;
		l |= *ptr++;
		fprintf(lst_fp, "-%04x", l); }
}

/*
 * Compare two symbol table entries
 */
compsym(sym1, sym2)
	char *sym1;
	char *sym2;
{
	int l, l1, l2;
	l1 = *sym1++ & SYMMASK;
	l2 = *sym2++ & SYMMASK;
	l = (l1 < l2) ? l1 : l2;
	do {
		if(*sym1 > *sym2)
			return 1;
		if(*sym1++ < *sym2++)
			return -1; }
	while(--l);
	if(l1 > l2)
		return 1;
	if(l1 < l2)
		return -1;
	return 0;
}

/*
 * Convert character to upper case if NOT case sensitive
 */
chupper(c)
	char c;
{
	return casf ? c : ((c >= 'a') && (c <= 'z')) ? c - ('a'-'A') : c;
}

/*
 * Open a filename with the appriopriate extension &
 * report an error if not possible
 */
FILE *open_file(filename, extension, options)
	char *filename, *extension, *options;
{
	char buffer[100], *ptr, *dot;
	FILE *fp;

	dot = 0;

	for(ptr = buffer; *ptr = *filename; ++ptr) {	/* Copy filename */
		if(*filename == '.')
			dot = filename;
		else if(*filename == '\\')
			dot = 0;
		++filename; }

	if(!dot) {									/* No extension supplied */
		*ptr++ = '.';
		do
			*ptr++ = *extension;
		while(*extension++); }
	else
		*dot = 0;

	if(!(fp = fopen(buffer, options))) {
		fprintf(stderr,"Unable to access: '%s'\n", buffer);
		exit(-1); }

	return fp;
}

/*
 * Main program
 */
main(argc, argv)
	unsigned argc;
	char *argv[];
{
	unsigned temp, i, daddr;
	char c, *ptr, *lfile, *cfile;

	if(argc < 2)
		abort("\nUse: asm85 <filename> [-cfiqst c=file l=file p=length w=width]\n\n?COPY.TXT 1983-2005 Dave Dunfield\n**See COPY.TXT**.\n");

/* parse for command line options. */
	lfile = cfile = argv[1];
	for(i=2; i < argc; ++i) {
		if(*(ptr = argv[i]) == '-') {		/* Enable switch */
			while(*++ptr) switch(toupper(*ptr)) {
				case 'C' : casf = -1;		break;
				case 'F' : fulf = -1;		break;
				case 'I' : intel = -1;		break;
				case 'Q' : quietf = -1;		break;
				case 'S' : symf = -1;		break;
				case 'T' : lfile = 0;		break;
				default: goto badopt; }
			continue; }
		if(*++ptr == '=') switch(toupper(*(ptr++ - 1))) {
			case 'L' : lfile = ptr;			continue;
			case 'C' : cfile = ptr;			continue;
			case 'P' : pagel=atoi(ptr)-1;	continue;
			case 'W' : pagew=atoi(ptr);		continue; }
	badopt:
		fprintf(stderr,"Invalid option: %s\n", argv[i]);
		exit(-1); }

	asm_fp = open_file(argv[1], "ASM", "r");
	hex_fp = open_file(cfile, "HEX", "w");
	lst_fp = (lfile) ? open_file(lfile, "LST", "w") : stdout;
	strncpy(title,argv[1], sizeof(title)-1);

/* first pass - build symbol table */
	if(!quietf) fprintf(stderr,"First pass... ");
	symtop = symtab;
	addr = ocnt = ecnt = 0; line = 1;
	while(readline()) {
		if(optr) {
			if(label[0]) {
				if(ptr = lookup_symbol(label)) {
					fprintf(lst_fp,"** Line %u - Duplicate symbol: '%s'\n",line,label);
					++ecnt; }
				else {
					if(!(ptr = create_symbol(label, addr))) {
						xabort("symbol");
						fprintf(lst_fp,"** Line %u - Symbol table overflow\n", line);
						++ecnt;
						break; } } }
			locins(instruction);
			emsg = 0;
			optr=operand;
			switch(type) {
				case 100 :		/* EQU statement */
					set_symbol(ptr, eval(operand));
					break;
				case 101 :		/* ORG statement */
					addr = eval(operand);
					break;
				case 102 :		/* DB statement */
					length=1;
					while(nxtelem()) ++length;
					break;
				case 103 :		/* DW statement */
					length=2;
					while(nxtelem()) length += 2;
					break;
				case 104 :		/* DS statement */
					addr += eval(operand);
					break;
				case 105 :		/* STR statement */
					c=operand[0];
					for(i=1;(operand[i])&&(c!=operand[i]); ++i)
						++length;
					if(opcode == 2)
						++length;
					break;
				case 106 :		/* END statement */
					goto end1;
				case 125 :		/* LINE directive */
					line = eval(operand); }
			if(emsg) {
				fprintf(lst_fp,"** Line %u - %s\n",line,etext[emsg]);
				++ecnt; } }
		else type = length = 0;
		++line; /* += (type < 120); */
		addr += length; }

/* second pass - generate code */
end1:
	if(!quietf) fprintf(stderr,"Second Pass... ");
	addr = emsg = 0;
	lcount = 9999; pcount = line = 1;
	rewind(asm_fp);
	while(readline()) {
		daddr = addr;
		if(optr) {
			if(!locins(instruction)) reperr(1); /* unknown opcode message */
			optr=operand;
			switch(type) {
				case 2 :		/* single byte immediate data */
					instruction[1]=eval(operand);
					break;
				case 3 :		/* single dest register */
					opcode = opcode|(getsreg(operand[0])<<3);
					break;
				case 4 :		/* register pair operand */
					opcode = opcode|(getdreg()<<4);
					break;
				case 5 :		/* source and dest registers */
					opcode=opcode|(getsreg(operand[0])<<3)|getsreg(operand[2]);
					if(operand[1] != ',') reperr(7);
					break;
				case 6 :		/* destination register, immediate data. */
					opcode=opcode|(getsreg(operand[0])<<3);
					if(operand[1] != ',') reperr(7);
					instruction[1]=eval(&operand[2]);
					break;
				case 7 :		/* register pair, immediate data */
					opcode=opcode|(getdreg()<<4);
					if(*optr++ != ',') reperr(7);
					instruction[1]=(temp=eval(optr));
					instruction[2]=temp >> 8;
					break;
				case 8 :		/* double byte immediate data */
					instruction[1]=(temp=eval(operand));
					instruction[2]=temp >> 8;
					break;
				case 9 :		/* single source register */
					opcode=opcode|getsreg(operand[0]);
					break;
				case 10 :		/* restart instruction */
					if(7<(temp=eval(operand))) reperr(2);
					opcode = opcode | (temp << 3);
					break;
				case 100 :		/* EQU statement */
					daddr = eval(operand);	/* for error messages */
					break;
				case 101 :		/* ORG statement */
					if(ocnt) wrrec();
					daddr=addr=eval(operand);
					break ;
				case 102 :		/* DB statement */
					optr=operand;
					do	instruction[length++]=eval(optr);
					while(*optr++ == ',');
					goto fixins;
				case 103 :		/* DW statement */
					optr=operand;
					do {
						temp = eval(optr);
						if(opcode) {
							instruction[length++]=temp;
							instruction[length++]=temp >> 8; }
						else {
							instruction[length++]=temp >> 8;
							instruction[length++]=temp; } }
					while(*optr++ == ',');
					goto fixins;
				case 104 :		/* DS statement */
					if(ocnt) wrrec();
					addr += eval(operand);
					break;
				case 105 :		/* STR statement */
					c=operand[0];
					for(i=1;((operand[i])&&(c!=operand[i]));++i)
						instruction[length++] = operand[i];
					if(!operand[i]) reperr(8);
					if(opcode == 1)			/* STRH */
						instruction[length-1] |= 0x80;
					else if(opcode == 2)	/* STRZ */
						instruction[length++] = 0;
					goto fixins;
				case 106 :		/* END statement */
					goto end2;
				case 120 :		/* PAGE statement */
					lcount=9999;
					break ;
				case 121 :		/* TITLE statement */
					strncpy(title, operand, sizeof(title)-1);
					break;
				case 122 :		/* SPACE directive */
					fprintf(lst_fp,"\n");
					++lcount;
					break ;
				case 123 :		/* LIST directive */
					if(nlist) --nlist ;
					break;
				case 124 :		/* NOLIST directive */
					++nlist;
					break;
				case 125 :		/* LINE directive */
					line = eval(operand); }
	instruction[0]=opcode;
fixins:	; }
	else type = length = 0;

/* generate formatted output listing */
	if(((type<120) && fulf && !nlist) || emsg) {
		if(++lcount >= pagel)
			write_title();
		fprintf(lst_fp,"%04x ",daddr);
		for (i=0 ; i < 6; ++i) {
			if(i<length)
				fprintf(lst_fp," %02x",instruction[i]);
			else
				fprintf(lst_fp,"   "); }
		fprintf(lst_fp," %c%5u  %s\n", (length <= 6) ? ' ': '+', line,buffer); }
	if(emsg) {
		fprintf(lst_fp,"  ** ERROR ** - %u - %s\n",emsg,etext[emsg]);
		++ecnt; ++lcount;
		emsg=0; }
	++line; /* += (type < 120); */
/* write output to code file */
	for(i=0; i<length; ++i) {
		if(!ocnt) {						/* starting new record */
			outrec[ocnt++]=addr>>8;
			outrec[ocnt++]=addr; }
		++addr;
		outrec[ocnt++]=instruction[i];
		if(ocnt>33) wrrec(); } }	/* record is full, write it out. */

end2:
	if(ocnt) wrrec();				/* write last partial record */
	if(intel) fprintf(hex_fp,":00000001FF\n");
	else	  fprintf(hex_fp,"S9030000FC\n");
	if(ecnt) fprintf(lst_fp,"\n %u error(s) occurred in this assembly.\n",ecnt);
	if(!quietf) fprintf(stderr,"%u error(s).\n",ecnt);

/* display symbol table */
	if(symf) {
		write_title();
		dump_symbols(); }

/* close files */
	fclose(asm_fp);
	fclose(hex_fp);
	fclose(lst_fp);

	return ecnt ? -2 : 0;
}

/***********************************************/
/* lookup instruction in the instruction table */
/***********************************************/
int locins(inst)
	char inst[];
{
	register char *ptr, *ptr1;

	ptr = opcodes;
	do {
		ptr1 = inst;
		while(*ptr == *ptr1) {
			++ptr;
			++ptr1; }
		if(((*ptr & 127) == *ptr1) && !*(ptr1+1)) {
			type = *++ptr;
			length = *++ptr;
			opcode = *++ptr;
			return 1; }
		while(*ptr++ >= 0); }
	while(*(ptr += 3));

	type = 119;
	return(inst[length = 0] == 0);
}

/*************************************************/
/* read a line from input file, and break it up. */
/*************************************************/
int readline()
{
	register int i;
	register char *ptr;

	if(!fgets(ptr = buffer, LINE_SIZE, asm_fp))
		return 0;

	i = 0;
	while(issymbol(*ptr)) {
		label[i] = chupper(*ptr++);
		if(i < SYMB_SIZE)
			++i; }
	label[i]=0;
	if(*ptr == ':')
		++ptr;
	while(isspace(*ptr))
		++ptr;
	if(((*ptr != '*') && (*ptr != ';')) || i) {
		i = 0;
		while(*ptr && !isspace(*ptr))
			instruction[i++] = toupper(*ptr++);
		instruction[i]=0;
		while(isspace(*ptr))
			++ptr;
		i = 0;
		while(*ptr && (i < 79))
			operand[i++] = *ptr++;
		(optr = operand)[i] = 0;
		return 1; }

	optr=0;
	return 1;
}

/*********************************************/
/* report an error, first error on line only */
/*********************************************/
reperr(en)
	int en;
{
	if (!emsg) emsg=en;
}

/**************************/
/* evaluate an expression */
/**************************/
int eval(cstr)
	char *cstr;
{
	register char c;
	unsigned result;

	optr=cstr;
	result=getval();
	while(!isterm(c=*optr)) {
		++optr;
		switch(c) {
			case '+' : result += getval();	break;
			case '-' : result -= getval();	break;
			case '*' : result *= getval();	break;
			case '/' : result /= getval();	break;
			case '\\': result %= getval();	break;
			case '&' : result &= getval();	break;
			case '|' : result |= getval();	break;
			case '^' : result ^= getval();	break;
			case '<' : result <<= getval();	break;
			case '>' : result >>= getval();	break;
			default: reperr(6); } }
	return result;
}

/***********************************************************/
/* get a value element from the expression being evaluated */
/***********************************************************/
int getval()
{
	unsigned i, b, val;
	char c, array[25], *ptr;

	switch(c = *optr++) {
		case '-' :
			return 0 - getval();
		case '~' :
			return ~getval();
		case '=' :
			i=getval();
			return (i<<8)+(i>>8);
		case '(' :
			val = eval(optr);
			if(*optr == ')')
				++optr;
			else
				reperr(6);
			return val; }
	--optr;

	val=b=i=0;
	switch(c) {
		case '$' :	b = 16;	goto getnum;
		case '@' :	b = 8;	goto getnum;
		case '%' :	b = 2;	goto getnum; }
	if(isdigit(c)) {
		array[i++] = c;
	getnum:
		while(isalnum(c = toupper(*++optr)))
			array[i++] = (c < 'A') ? c : c - 7;
		if((b == 16) && !i)
			val = addr;
		else {
			if(!b) {
				b = 10;
				switch(array[i-1]) {
					case 'H'-7 :	b = 16;	goto deci;
					case 'T'-7 :
					case 'D'-7 :	b = 10;	goto deci;
					case 'Q'-7 :
					case 'O'-7 :	b = 8;	goto deci;
					case 'B'-7 :	b = 2;
					deci:	--i;	} }
			if(!i)
				reperr(6);
			for(ptr = array; i; --i) {
				if((c = *ptr++ - '0') >= b)
					reperr(6);
				val = (val * b) + c; } } }
	else if(c==39) {					/* single quote. */
		++optr;
		while(((c=*optr++) != 39)&&c)
			val=(val << 8)+c;
		if(!c) reperr(8); }
	else if(c=='*') {					/* current program counter */
		++optr;
		val=addr; }
	else {								/* must be a label */
		i = 0;
		while(issymbol(c = chupper(*optr))) {
			++optr;
			label[i]=c;
			if(i < SYMB_SIZE)
				++i; }
		label[i]=0;
		if(!lookup_symbol(label)) reperr(5);
		val=value; }

	return val;
}

/***************************************/
/* test for valid terminator character */
/***************************************/
int isterm(c)
	char c;
{
	switch(c) {
		case 0 :
		case ' ' :
		case '\t':
		case ',' :
		case ')' :
			return 1; }
	return 0;
}

/*************************************/
/* Test for a valid symbol character */
/*************************************/
int issymbol(c)
	char c;
{
	switch(c) {			/* Special allowed characters */
		case '_' :
		case '?' :
		case '!' :
		case '.' :
			return 1; }

	return isalnum(c);
}

/***************************************************/
/* get code value for an 8 bit register identifier */
/***************************************************/
int getsreg(c)
	char c;
{
	register int i;
	static char sregs[] = { "BCDEHLMA" } ;

	c=chupper(c);
	for(i=0; i < 8; ++i)
		if(sregs[i] == c)
			return i;
	reperr(4);
}

/***************************************************/
/* get code value for a 16 bit register identifier */
/***************************************************/
int getdreg()
{
	switch(chupper(*optr++)) {
		case 'B' :	return 0;
		case 'D' :	return 1;
		case 'H' :	return 2;
		case 'P' :
			if((chupper(*optr++) == 'S') && (chupper(*optr++) == 'W'))
				return 3;
			break;
		case 'S' :
			if(chupper(*optr++) == 'P')
				return 3; }
	reperr(4);
	return 0;
}

/*********************************************/
/* skip to next element in the argument list */
/*********************************************/
int nxtelem()
{
	register char c;
	while((c=*optr)&&(c!=' ')&&(c!=9)) {
		++optr;
		if(c==39) {
			while((c=*optr)&&(c!=39)) ++optr;
			++optr; }
		if(c==',')
			return(1); }
		return(0);
}

/*******************************/
/* write record to output file */
/*******************************/
wrrec()
{
	register unsigned i, chk, c;

	xhex(ocnt-2);
	if(intel) {					/* intel hex format */
		chk = outrec[0] + outrec[1] + ocnt - 2;
		fprintf(hex_fp,":%02x%02x%02x00", ocnt-2, outrec[0], outrec[1]);
		for(i=2; i<ocnt; ++i) {
			fprintf(hex_fp,"%02x", c = outrec[i]);
			chk += c; }
		fprintf(hex_fp,"%02x\n", 255 & (0-chk)); }
	else {						/* motorola hex format */
		chk = ocnt + 1;
		fprintf(hex_fp,"S1%02x", ocnt + 1);
		for(i=0; i<ocnt; ++i) {
			fprintf(hex_fp,"%02x", c = outrec[i]);
			chk += c; }
		fprintf(hex_fp,"%02x\n", 255 & ~chk); }
	ocnt = 0;
}

/*
 * Write out title
 */
write_title()
{
	int w;
	char *ptr;

	if(pcount > 1)
		putc('\f',lst_fp);
	lcount = 1;
	fprintf(lst_fp,"DUNFIELD 8085 ASSEMBLER: ");
	ptr = title;
	for(w = 35; w < pagew; ++w)
		putc(*ptr ? *ptr++ : ' ', lst_fp);
	fprintf(lst_fp,"PAGE: %u\n\n", pcount);
	++pcount;
}
