/*
 * 8096 Cross assembler
 *
 * ?COPY.TXT 1983-2005 Dave Dunfield
 * **See COPY.TXT**.
 */
#include <stdio.h>
#include <ctype.h>

#include "xasm.h"

/* Bits in flag byte of instruction lookup table */
#define	PREFIX		0x80	/* Instructions has prefix byte */
#define	BYTE		0x01	/* Operands are bytes */
#define	OPR3		0x02	/* Instruction supports three operands */
#define	WORD1		0x04	/* First operand must be word register */

/* Operand types returned by memop() */
#define	REGISTER	0x00	/* Register operand */
#define	IMMEDIATE	0x01	/* Immediate operand */
#define	INDIRECT	0x02	/* Indirect through register */
#define	INDEXED		0x03	/* Indirect with offset */

/* 8096 opcode table */
static char opcodes[] = {
/* Type 1 - No operands */
	'C','L','R','C',	0x81, 0xF8, 0x00,
	'C','L','R','V','T',0x81, 0xFC, 0x00,
	'D','I',			0x81, 0xFA, 0x00,
	'E','I',			0x81, 0xFB, 0x00,
	'N','O','P',		0x81, 0xFD, 0x00,
	'R','E','T',		0x81, 0xF0, 0x00,
	'R','S','T',		0x81, 0xFF, 0x00,
	'S','E','T','C',	0x81, 0xF9, 0x00,
	'T','R','A','P',	0x81, 0xF7, 0x00,
	'P','O','P','F',	0x81, 0xF3, 0x00,
	'P','U','S','H','F',0x81, 0xF2, 0x00,
	'P','U','S','H','A',0x81, 0xF4, 0x00,
	'P','O','P','A',	0x81, 0xF5, 0x00,
/* Type 2 - single register */
	'B','R',			0x82, 0xE3, 0x00,
	'C','L','R',		0x82, 0x01, 0x00,
	'D','E','C',		0x82, 0x05, 0x00,
	'E','X','T',		0x82, 0x06, 0x00,
	'E','X','T','B',	0x82, 0x16, 0x00,
	'I','N','C',		0x82, 0x07, 0x00,
	'N','E','G',		0x82, 0x03, 0x00,
	'N','O','T',		0x82, 0x02, 0x00,
	'S','K','I','P',	0x82, 0x00, 0x00,
	'C','L','R','B',	0x82, 0x11, 0x01,
	'D','E','C','B',	0x82, 0x15, 0x01,
	'I','N','C','B',	0x82, 0x17, 0x01,
	'N','E','G','B',	0x82, 0x13, 0x01,
	'N','O','T','B',	0x82, 0x12, 0x01,
/* Type 3 - single memory operand */
	'P','O','P',		0x83, 0xCC, 0x00,
	'P','U','S','H',	0x83, 0xC8, 0x00,
/* Type 4 - Two/three operand (reg/reg, aop) */
	'D','I','V',		0x84, 0x8C, 0x80,
	'A','D','D','C',	0x84, 0xA4, 0x00,
	'C','M','P',		0x84, 0x88, 0x00,
	'D','I','V','U',	0x84, 0x8C, 0x00,
	'L','D',			0x84, 0xA0, 0x00,
	'O','R',			0x84, 0x80, 0x00,
	'S','T',			0x84, 0xC0, 0x00,
	'S','U','B','C',	0x84, 0xA8, 0x00,
	'X','O','R',		0x84, 0x84, 0x00,
	'A','D','D','C','B',0x84, 0xB4, 0x01,
	'C','M','P','B',	0x84, 0x98, 0x01,
	'L','D','B',		0x84, 0xB0, 0x01,
	'O','R','B',		0x84, 0x90, 0x01,
	'S','T','B',		0x84, 0xC4, 0x01,
	'S','U','B','C','B',0x84, 0xB8, 0x01,
	'X','O','R','B',	0x84, 0x94, 0x01,
	'A','D','D',		0x84, 0x64, 0x02,
	'A','N','D',		0x84, 0x60, 0x02,
	'M','U','L',		0x84, 0x6C, 0x86,
	'M','U','L','U',	0x84, 0x6C, 0x06,
	'S','U','B',		0x84, 0x68, 0x02,
	'A','D','D','B',	0x84, 0x74, 0x03,
	'A','N','D','B',	0x84, 0x70, 0x03,
	'M','U','L','B',	0x84, 0x7C, 0x87,
	'M','U','L','U','B',0x84, 0x7C, 0x07,
	'S','U','B','B',	0x84, 0x78, 0x03,
	'D','I','V','B',	0x84, 0x9C, 0x85,
	'D','I','V','U','B',0x84, 0x9C, 0x05,
	'L','D','B','S','E',0x84, 0xBC, 0x05,
	'L','D','B','Z','E',0x84, 0xAC, 0x05,
/* Type 5 - Registers only */
	'N','O','R','M','L',0x85, 0x0F, 0x05,
/* Type 6 - 16 bit displacement */
	'L','C','A','L','L',0x86, 0xEF, 0x00,
	'L','J','M','P',	0x86, 0xE7, 0x00,
/* Type 7 - 8 bit displacement */
	'J','C',			0x87, 0xDB, 0x00,
	'J','E',			0x87, 0xDF, 0x00,
	'J','G','E',		0x87, 0xD6, 0x00,
	'J','G','T',		0x87, 0xD2, 0x00,
	'J','H',			0x87, 0xD9, 0x00,
	'J','L','E',		0x87, 0xDA, 0x00,
	'J','L','T',		0x87, 0xDE, 0x00,
	'J','N','C',		0x87, 0xD3, 0x00,
	'J','N','E',		0x87, 0xD7, 0x00,
	'J','N','H',		0x87, 0xD1, 0x00,
	'J','N','S','T',	0x87, 0xD0, 0x00,
	'J','N','V',		0x87, 0xD5, 0x00,
	'J','N','V','T',	0x87, 0xD4, 0x00,
	'J','S','T',		0x87, 0xD8, 0x00,
	'J','V',			0x87, 0xDD, 0x00,
	'J','V','T',		0x87, 0xDC, 0x00,
/* Type 8 - 11 bit displacement */
	'S','C','A','L','L',0x88, 0x28, 0x00,
	'S','J','M','P',	0x88, 0x20, 0x00,
/* Type 9 - breg, cadd */
	'D','J','N','Z',	0x89, 0xE0, 0x01,
/* Type 10 - breg, bitno, cadd */
	'J','B','C',		0x8A, 0x30, 0x01,
	'J','B','S',		0x8A, 0x38, 0x01,
/* Type 11 - reg,breg or reg,#count */
	'S','H','L',		0x8B, 0x09, 0x00,
	'S','H','L','B',	0x8B, 0x19, 0x01,
	'S','H','L','L',	0x8B, 0x0D, 0x00,
	'S','H','R',		0x8B, 0x08, 0x00,
	'S','H','R','B',	0x8B, 0x18, 0x01,
	'S','H','R','L',	0x8B, 0x0C, 0x00,
	'S','H','R','A',	0x8B, 0x0A, 0x00,
	'S','H','R','A','B',0x8B, 0x1A, 0x01,
	'S','H','R','A','L',0x8B, 0x0E, 0x00,
/* Directives */
	'E','Q','U',		228,	0, 0x00,
	'O','R','G',		229,	0, 0x00,
	'D','B',			230,	0, 0x00,
	'D','W',			231,	-1,0x00,
	'D','R','W',		231,	0, 0x00,
	'S','T','R',		232,	0, 0x00,
	'S','T','R','H',	232,	1, 0x00,
	'S','T','R','Z',	232,	2, 0x00,
	'D','S',			233,	0, 0x00,
	'F','C','B',		230,	0, 0x00,
	'F','D','B',		231,	-1,0x00,
	'R','D','B',		231,	0, 0x00,
	'F','C','C',		232,	0, 0x00,
	'F','C','C','H',	232,	1, 0x00,
	'F','C','C','Z',	232,	2, 0x00,
	'R','M','B',		233,	0, 0x00,
	'E','N','D',		234,	0, 0x00,
	'P','A','G','E',	248,	0, 0x00,
	'T','I','T','L','E',249,	0, 0x00,
	'S','P','A','C','E',250,	0, 0x00,
	'L','I','S','T',	251,	0, 0x00,
	'N','O','L','I','S','T',252,0, 0x00,
	'L','I','N','E',	253,	0, 0x00,
	0 };	/* end of table */

/* Internal register and value table
char *intreg[] = {
	"SP", "AX", "BX", "CX", "DX", "AL", "BL", "CL", "DL", "AH", "BH", "CH", "DH",
	0 };
int intval[] = {
	0x24, 0x26, 0x28, 0x2A, 0x2C, 0x26, 0x28, 0x2A, 0x2C, 0x27, 0x29, 0x2B, 0x2D
	}; */

/* error messages */
static char *etext[] = { "?",
	"Unknown instruction",
	"Out of range",
	"Invalid addressing mode",
	"Improper alignment",
	"Undefined symbol",
	"Invalid expression syntax",
	"Invalid argument format",
	"Improperly delimited string" } ;

/* Symbol table & free pointer */
char symtab[SYMB_POOL], *symtop;

/* misc. global variables */
char buffer[LINE_SIZE+1], label[SYMB_SIZE+1], title[50], operand[80], optr,
	itype, opcode, oflags;
unsigned char instruction[80], outrec[35];

char optf=3, symf=0, intel=0, fulf=0, quietf=0, nlist=0, casf=0;

unsigned addr, line=0, pcount=0, lcount=0, value, length, emsg, ecnt,
	ocnt=0, pagel=59, pagew=80;

unsigned operands[3][4];

FILE *asm_fp, *hex_fp, *lst_fp;

/*
 * Define a symbol in the symbol table
 */
char *create_symbol(symbol, value)
	char *symbol;
	int value;
{
	register char *ptr, *ptr1;

	ptr = ptr1 = symtop;
	while(*symbol)
		*++ptr = *symbol++;
	*ptr1 = ptr - ptr1;
	*++ptr = value >> 8;
	*++ptr = value;
	if(ptr < (symtab+sizeof(symtab))) {
		symtop = ptr + 1;
		return ptr1; }
	return 0;
}

/*
 * Lookup a symbol in the symbol table
 */
char *lookup_symbol(symbol)
	char *symbol;
{
	register int l;
	register unsigned char *ptr, *ptr1;
	char *ptr2;

	ptr = symtab;
again:
	if((ptr2 = ptr) >= symtop) {
		value = 0x8888;
		return 0; }
	ptr1 = symbol;
	l = *ptr++ & SYMMASK;
	while(l--) {
		if(*ptr1++ != *ptr++) {
			ptr += l + 2;
			goto again; } }
	if(*ptr1) {
		ptr += 2;
		goto again; }
	value = *ptr++ << 8;
	value |= *ptr;
	return ptr2;
}

/*
 * Set the value for a symbol in the symbol table
 */
set_symbol(ptr, value)
	char *ptr;
	unsigned value;
{
	ptr += (*ptr & SYMMASK);
	*++ptr = value >> 8;
	*++ptr = value;
}

/*
 * Display the symbol table (sorted)
 */
dump_symbols()
{
	unsigned char *ptr, *hptr;
	unsigned int l, h, w;

	fprintf(lst_fp, "SYMBOL TABLE:\n\n");
	w = 0;
	for(;;) {
		for(hptr = symtab; hptr < symtop; hptr += (*hptr & SYMMASK) + 3)
			if(!(*hptr & SYMSORT))
				goto found;
		putc('\n', lst_fp);
		return;
found:	for(ptr = (*hptr & SYMMASK) + hptr + 3; ptr < symtop; ptr += (*ptr & SYMMASK) + 3) {
			if(*ptr & SYMSORT)
				continue;
			if(compsym(ptr, hptr) < 0)
				hptr = ptr; }
		*(ptr = hptr) |= SYMSORT;
		h = l = *ptr++ & SYMMASK;
		if((w + l + 7) >= pagew) {			/* Overrun page length */
			putc('\n', lst_fp);
			w = 0; }
		if(w) {								/* Not a start of line - separate */
			fprintf(lst_fp, "   ");
			w += 3; }
		while(l--)
			putc(*ptr++, lst_fp);
		w += (l = (h > 8) ? 24 : 8) + 5;	/* Calculate extended length */
		while(h++ < l)
			putc(' ', lst_fp);
		l = *ptr++ << 8;
		l |= *ptr++;
		fprintf(lst_fp, "-%04x", l); }
}

/*
 * Compare two symbol table entries
 */
compsym(sym1, sym2)
	char *sym1;
	char *sym2;
{
	int l, l1, l2;
	l1 = *sym1++ & SYMMASK;
	l2 = *sym2++ & SYMMASK;
	l = (l1 < l2) ? l1 : l2;
	do {
		if(*sym1 > *sym2)
			return 1;
		if(*sym1++ < *sym2++)
			return -1; }
	while(--l);
	if(l1 > l2)
		return 1;
	if(l1 < l2)
		return -1;
	return 0;
}

/*
 * Convert character to upper case if NOT case sensitive
 */
chupper(c)
	char c;
{
	return casf ? c : ((c >= 'a') && (c <= 'z')) ? c - ('a'-'A') : c;
}

/*
 * Open a filename with the appriopriate extension &
 * report an error if not possible
 */
FILE *open_file(filename, extension, options)
	char *filename, *extension, *options;
{
	char buffer[100], *ptr, *dot;
	FILE *fp;

	dot = 0;

	for(ptr = buffer; *ptr = *filename; ++ptr) {	/* Copy filename */
		if(*filename == '.')
			dot = filename;
		else if(*filename == '\\')
			dot = 0;
		++filename; }

	if(!dot) {									/* No extension supplied */
		*ptr++ = '.';
		do
			*ptr++ = *extension;
		while(*extension++); }
	else
		*dot = 0;

	if(!(fp = fopen(buffer, options))) {
		fprintf(stderr,"Unable to access: '%s'\n", buffer);
		exit(-1); }

	return fp;
}

/*
 * Main program
 */
main(argc, argv)
	unsigned argc;
	char *argv[];
{	unsigned i, temp, daddr, vbyt;
	int stemp;
	char chr, cflg, pflg, *ptr, *lfile, *cfile;

	if(argc < 2)
		abort("\nUse: asm96 <filename> [-cfiqst c=file l=file o=n p=length w=width]\n\n?COPY.TXT 1983-2005 Dave Dunfield\n**See COPY.TXT**.\n");

	pflg = 0;

/* parse for command line options. */
	lfile = cfile = argv[1];
	for(i=2; i < argc; ++i) {
		if(*(ptr = argv[i]) == '-') {		/* Enable switch */
			while(*++ptr) switch(toupper(*ptr)) {
				case 'C' : casf = -1;		break;
				case 'F' : fulf = -1;		break;
				case 'I' : intel = -1;		break;
				case 'Q' : quietf = -1;		break;
				case 'S' : symf = -1;		break;
				case 'T' : lfile = 0;		break;
				default: goto badopt; }
			continue; }
		if(*++ptr == '=') switch(toupper(*(ptr++ - 1))) {
			case 'L' : lfile = ptr;			continue;
			case 'C' : cfile = ptr;			continue;
			case 'O' : optf=atoi(ptr);		continue;
			case 'P' : pagel=atoi(ptr)-1;	continue;
			case 'W' : pagew=atoi(ptr);		continue; }
	badopt:
		fprintf(stderr,"Invalid option: %s\n", argv[i]);
		exit(-1); }

	asm_fp = open_file(argv[1], "ASM", "r");
	hex_fp = open_file(cfile, "HEX", "w");
	lst_fp = (lfile) ? open_file(lfile, "LST", "w") : stdout;
	strncpy(title, argv[1], sizeof(title)-1);

/* first pass - build symbol table */
	symtop = symtab;
	do {
		addr = cflg = 0; line = 1;
		if(!quietf) if(pflg) fprintf(stderr,"Opt... ");
			else	fprintf(stderr,"First pass... ");
		while(readline()) {
			if(!optr) {
				locins(instruction);
				if(label[0]) {				/* label exists */
					if(pflg) {				/* optomization pass */
						ptr = lookup_symbol(label);
						if(itype==100) {	/* EQU statement */
							optr = 0; vbyt = value;
							if((temp=eval()) != vbyt) {
								if(pflg >= optf) optfail();
								set_symbol(ptr, temp);
								cflg=1; } }
						else if(value != addr) {
							if(pflg >= optf) optfail();
							set_symbol(ptr, addr);
							cflg=1; } }
					else {							/* original pass-1 */
						if(lookup_symbol(label)) {	/* duplicate label */
							fprintf(lst_fp,"** Line %u - Duplicate symbol: '%s'\n",line,label);
							++ecnt; }
						else {
							cflg=1;			/* indicate symbol table change */
							if(!(ptr = create_symbol(label, addr))) {
								xabort("symbol");
								fprintf(lst_fp,"** Line %u - Symbol table overflow\n", line);
								++ecnt;
								break; } } } }
				emsg = length = optr = 0;
				switch(itype) {
					case 1 :		/* inherent addressing */
						length = 1;
						break;
					case 2 :		/* Register only */
						length = memop(operands, 0) + 1;
						break;
					case 3 :		/* Memory operand only */
						length = memop(operands, 0) + 1;
						break;
					case 4 :		/* Two/three operands */
						length = memop(operands, 0) + 1 + ((oflags & PREFIX) != 0);
						test_next(',');
						length += memop(operands, 0);
						if(test_next(','))
							length += memop(operands, 0);
						break;
					case 5 :		/* Registers only */
					case 6 :		/* 16 bit transfers */
					case 9 :		/* breg, caddr */
					case 10 :		/* breg, bit, disp */
					case 11 :		/* Shifts */
						length = 3;
						break;
					case 7 :		/* 8 bit transfers */
					case 8 :		/* 11 bit transfers */
						length = 2;
						break;
					case 100 :			/* EQU directive */
						if(!pflg) {
							set_symbol(ptr, eval());
						err1st:
							if(emsg && !optf) {
								fprintf(lst_fp,"** Line %u - %s\n",line,etext[emsg]);
								++ecnt; } }
						break;
					case 101 :			/* ORG directive */
						addr=eval();
						goto err1st;
					case 102 :			/* FCB statement */
						length=1;
						while(nxtelem()) ++length;
						break;
					case 103 :			/* FDB statement */
						length=2;
						while(nxtelem()) length += 2;
						break;
					case 104 :			/* FCC statements */
						chr=operand[0];
						for(i=1;(operand[i])&&(chr!=operand[i]); ++i) ++length;
						if(opcode == 2) ++length;
						break;
					case 105 :			/* RMB statement */
						addr = addr + eval();
						goto err1st;
					case 106 :			/* END statement */
						goto end1;
					case 125 :			/* LINE directive */
						line = eval(); } }
			else
				length = 0;
			addr += length;
			++line; /* += (itype < 120); */ }
end1:	++pflg;
		rewind(asm_fp); }
	while((optf)&&(cflg)&&(!ecnt));

/* second pass - generate object code */

	if(!quietf)
		fprintf(stderr,"Second pass... ");
	addr = emsg = 0; line = pcount = 1; lcount = 9999;
	while(readline()) {
		daddr = addr;
		if(!optr) {
			if(!locins(instruction))
				reperr(1);			/* unknown opcode */
			length = temp = optr = 0;
			switch(itype) {
				case 1 :		/* inherent addressing */
					instruction[length++] = opcode;
					break;
				case 2 :		/* Single Register only */
					length = memop(operands[0], 0x0E) + 1;
					instruction[0] = opcode;
					write_operand(operands[0], 1);
					break;
				case 3 :		/* Memory operand */
					length = memop(operands[0], 0) + 1;
					instruction[0] = opcode | *operands[0];
					write_operand(operands[0], 1);
					break;
				case 4 :		/* Two/Three operands */
					length = memop(operands[temp = 0], 0x0E) + 1;
					if(oflags & PREFIX) {
						instruction[temp++] = 0xFE;
						++length; }
					expect(',');
					if((oflags & WORD1) && (operands[0][2] & 0x01))
						reperr(4);
					length += memop(operands[1], 0);
					if(test_next(',')) {
						if(!(oflags & OPR3))
							reperr(3);
						if(*operands[1])
							reperr(3);
						length += memop(operands[2], 0);
						instruction[temp++] = *operands[2] | (opcode & 0xDF);
						temp = write_operand(operands[2], temp); }
					else
						instruction[temp++] = opcode | *operands[1];
					temp = write_operand(operands[1], temp);
					write_operand(operands[0], temp);
					break;
				case 5 :			/* Two registers only */
					length = memop(operands[0], 0x0E) + 1;
					expect(',');
					length += memop(operands[1], 0x0E);
					if((oflags & WORD1) && (operands[0][2] & 0x01))
						reperr(4);
					instruction[0] = opcode;
					write_operand(operands[1], 1);
					write_operand(operands[0], 2);
					break;
				case 6 :			/* 16 bit transfers */
					length = 3;
					temp = (eval() - addr) - 3;
					instruction[0] = opcode;
					instruction[1] = temp;
					instruction[2] = temp >> 8;
					break;
				case 7 :			/* 8 bit transfers */
					length = 2;
					stemp = temp = (eval() - addr) - 2;
					if((stemp > 127) || (stemp < -128))
						reperr(2);
					instruction[0] = opcode;
					instruction[1] = temp;
					break;
				case 8 :			/* 11 bit transfers */
					length = 2;
					stemp = temp = (eval() - addr) - 2;
					if((stemp > 1023) || (stemp < -1024))
						reperr(2);
					instruction[0] = opcode | ((temp >> 8) & 0x07);
					instruction[1] = temp;
					break;
				case 9 :			/* breg,caddr */
					length = memop(operands[0], 0xE0) + 2;
				dojmp8r:
					expect(',');
					stemp = temp = (eval() - addr) - 3;
					if((stemp > 127) || (stemp < -128))
						reperr(2);
					instruction[0] = opcode;
					write_operand(operands[0], 1);
					instruction[2] = temp;
					break;
				case 10 :			/* breg,bit,caddr */
					length = memop(operands[0], 0xE0) + 2;
					expect(',');
					if((temp = eval()) > 7)
						reperr(2);
					opcode |= temp;
					goto dojmp8r;
				case 11 :			/* Shifts */
					length = memop(operands[0], 0x0E) + 2;
					expect(',');
					memop(operands[1], 0x0C);
					if(operands[1][0]) {
						if(operands[1][2] & 0xFFF0)
							reperr(2); }
					else if(operands[1][2] < 24)
						reperr(2);
					instruction[0] = opcode;
					write_operand(operands[1], 1);
					write_operand(operands[0], 2);
					break;
				case 100 :			/* equate statement */
					daddr=eval();	/* generate errors if any */
					break;
				case 101 :			/* ORG statement */
					if(ocnt) wrrec();
					daddr=addr=eval();
					break;
				case 102 :			/* FCB statement */
					do {
						instruction[length++] = eval(); }
					while(test_next(','));
					break;
				case 103 :			/* FDB statement */
					do {
						temp = eval();
						if(opcode) {
							instruction[length++] = temp;
							instruction[length++] = temp >> 8; }
						else {
							instruction[length++] = temp >> 8;
							instruction[length++] = temp; }}
					while(test_next(','));
					break;
				case 104 :			/* FCC statements */
					chr=operand[0];
					for(i=1;((operand[i])&&(chr!=operand[i])); ++i)
						instruction[length++] = operand[i];
					if(!operand[i])	reperr(8);
					if(opcode == 1)
						instruction[length-1] = instruction[length-1] | 0x80;
					if(opcode == 2)
						instruction[length++]=0;
					break;
				case 105 :			/* RMB statement */
					if(ocnt) wrrec();
					addr += eval();
					break;
				case 106 :			/* END statement */
					goto end2;
				case 120 :			/* PAGE statement */
					lcount=9999;
					break;
				case 121 :			/* TITLE directive */
					strncpy(title, operand, sizeof(title)-1);
					break;
				case 122 :			/* SPACE directive */
					fprintf(lst_fp,"\n");
					++lcount;
					break;
				case 123 :			/* LIST directive */
					if(nlist) --nlist;
					break;
				case 124 :			/* NOLIST directive */
					++nlist;
					break;
				case 125 :			/* LINE directive */
					line = eval(); } }
		else
			length=itype=0;

/* generate listing */
	if(((itype<120) && fulf && !nlist) || emsg) {
		if(++lcount >= pagel)
			write_title();
		fprintf(lst_fp,"%04x ",daddr);
			for(i=0; i < 6; ++i) {
				if(i < length)
					fprintf(lst_fp," %02x",instruction[i]);
				else
					fprintf(lst_fp,"   "); }
			fprintf(lst_fp," %c%5u  %s\n",(length <= 6) ? ' ' : '+',line,buffer); }
		if(emsg) {
			fprintf(lst_fp,"  ** ERROR ** - %u - %s\n", emsg, etext[emsg]);
			++ecnt; ++lcount;
			emsg=0; }
		++line; /* += (itype<120); */
/* write code to output record */
	for(i=0; i<length; ++i) {
		if(!ocnt) {			/* first byte of record */
			outrec[ocnt++]=addr>>8;
			outrec[ocnt++]=addr; }
		++addr;
		outrec[ocnt++]=instruction[i];
		if(ocnt>33) wrrec(); } }	/* record is full, write it */

end2:
	if(ocnt) wrrec();
	if(intel) fprintf(hex_fp,":00000001FF\n");
	else fprintf(hex_fp,"S9030000FC\n");

	if(ecnt) fprintf(lst_fp,"\n %u error(s) occurred in this assembly.\n",ecnt);
	if(!quietf) fprintf(stderr,"%u error(s).\n",ecnt);

/* display the symbol table */
	if(symf) {
		write_title();
		dump_symbols(); }

	fclose(asm_fp);
	fclose(hex_fp);
	fclose(lst_fp);

	return ecnt ? -2 : 0;
}

/***********************************************/
/* lookup instruction in the instruction table */
/***********************************************/
int locins(inst)
	char inst[];
{
	register char *ptr, *ptr1;

	ptr = opcodes;
	do {
		ptr1 = inst;
		while(*ptr == *ptr1) {
			++ptr;
			++ptr1; }
		if((*ptr < 0) && !*ptr1) {
			itype = *ptr++ & 127;
			opcode = *ptr++;
			oflags = *ptr;
			return 1; }
		while(*ptr > 0)
			++ptr; }
	while(*(ptr += 3));

	itype = 119;
	return(inst[0] == 0);
}

/*************************************************/
/* read a line from input file, and break it up. */
/*************************************************/
int readline()
{
	register int i;
	register char *ptr;

	if(!fgets(ptr = buffer, LINE_SIZE, asm_fp))
		return 0;

	i = 0;
	while(issymbol(*ptr)) {
		label[i] = chupper(*ptr++);
		if(i < SYMB_SIZE)
			++i; }
	label[i]=0;
	if(*ptr == ':')
		++ptr;
	while(isspace(*ptr))
		++ptr;
	if(((*ptr != '*') && (*ptr != ';')) || i) {
		i = 0;
		while(*ptr && !isspace(*ptr))
			instruction[i++] = toupper(*ptr++);
		instruction[i]=0;
		while(isspace(*ptr))
			++ptr;
		optr = i = 0;
		while(*ptr && (i < 79))
			operand[i++] = *ptr++;
		operand[i] = 0;
		return 1; }

	return optr = 1;
}

/***************************************/
/* test for valid terminator character */
/***************************************/
int isterm(chr)
	register char chr;
{
	switch(chr) {
		case 0 :
		case ' ' :
		case '\t' :
		case ',' :
		case '[' :
		case ']' :
		case ')' :
			return 1; }
	return 0;
}

/*************************************/
/* Test for a valid symbol character */
/*************************************/
int issymbol(c)
	char c;
{
	switch(c) {			/* Special allowed characters */
		case '_' :
		case '?' :
		case '!' :
		case '.' :
			return 1; }

	return isalnum(c);
}

/* report an error */
reperr(errn)
	register int errn;
{
	if(!emsg) emsg=errn;
}

/*******************************/
/* write record to output file */
/*******************************/
wrrec()
{	register unsigned i, chk, chr;

	xhex(ocnt-2);
	if(intel) {					/* intel hex format */
		chk = outrec[0] + outrec[1] + ocnt - 2;
		fprintf(hex_fp,":%02x%02x%02x00", ocnt-2, outrec[0], outrec[1]);
		for(i=2; i<ocnt; ++i) {
			fprintf(hex_fp,"%02x", chr = outrec[i]);
			chk += chr; }
		fprintf(hex_fp,"%02x\n", 255 & (0-chk)); }
	else {						/* motorola hex format */
		chk = ocnt + 1;
		fprintf(hex_fp,"S1%02x", ocnt + 1);
		for(i=0; i<ocnt; ++i) {
			fprintf(hex_fp,"%02x", chr = outrec[i]);
			chk += chr; }
		fprintf(hex_fp,"%02x\n", 255 & ~chk); }
	ocnt = 0;
}

/*
 * Test that the next character in the expression is one expected
 */
test_next(c)
	char c;
{
	if(operand[optr] == c) {
		++optr;
		return -1; }
	return 0;
}

/*
 * Expect the following symbol, otherwise error
 */
expect(c)
	char c;
{
	if(!test_next(c))
		reperr(6);
}

/*
 * Evaluate an expression.
 */
int eval()
{	register char c;
	unsigned result;

	result=getval();
	while(!isterm(c = operand[optr])) {
		++optr;
		switch(c) {
			case '+' : result += getval();	break;
			case '-' : result -= getval();	break;
			case '*' : result *= getval();	break;
			case '/' : result /= getval();	break;
			case '\\': result %= getval();	break;
			case '&' : result &= getval();	break;
			case '|' : result |= getval();	break;
			case '^' : result ^= getval();	break;
			case '<' : result <<= getval();	break;
			case '>' : result >>= getval();	break;
			default: reperr(6); } }

	return result;
}

/*
 * Get a single value (number or symbol)
 */
int getval()
{
	unsigned i, b, val;
	char chr, array[25], *ptr;

	switch(chr = operand[optr++]) {
		case '-' :				/* Negated value */
			return 0 - getval();
		case '~' :				/* Complemented value */
			return ~getval();
		case '=' :				/* Swap high and low bytes */
			i=getval();
			return (i<<8)+(i>>8);
		case '(' :				/* Nested expression */
			val = eval();
			if(!test_next(')'))
				reperr(6);
			return val; }
	--optr;

	val=b=i=0;
	switch(chr) {
		case '$' :	b = 16;	goto getnum;
		case '@' :	b = 8;	goto getnum;
		case '%' :	b = 2;	goto getnum; }
	if(isdigit(chr)) {
		array[i++] = chr;
	getnum:
		while(isalnum(chr = toupper(operand[++optr])))
			array[i++] = (chr < 'A') ? chr : chr - 7;
		if((b == 16) && !i)
			val = addr;
		else {
			if(!b) {
				b = 10;
				switch(array[i-1]) {
					case 'H'-7 :	b = 16;	goto deci;
					case 'T'-7 :
					case 'D'-7 :	b = 10;	goto deci;
					case 'Q'-7 :
					case 'O'-7 :	b = 8;	goto deci;
					case 'B'-7 :	b = 2;
					deci:	--i;	} }
			if(!i)
				reperr(6);
			for(ptr = array; i; --i) {
				if((chr = *ptr++ - '0') >= b)
					reperr(6);
				val = (val * b) + chr; } } }
	else if(chr=='\'') {				/* Single quote. */
		++optr;
		while(((chr=operand[optr++]) != '\'')&& chr)
			val=(val << 8)+chr;
		if(!chr) reperr(8); }
	else if(chr=='*') {					/* Program counter */
		++optr;
		val=addr; }
	else {								/* Must be a label */
		i = 0;
		while(issymbol(chr = chupper(operand[optr]))) {
			++optr;
			label[i]=chr;
			if(i < SYMB_SIZE)
				++i; }
		label[i]=0;

/*		for(i=0; intreg[i]; ++i) {
			if(!strcmp(intreg[i], label))
				return intval[i]; }
*/
		if(!lookup_symbol(label)) reperr(5);
		val=value; }

	return val;
}

/* evaluate memory operand, and determine type of */
/* addressing, and size of instruction.           */
int memop(op, disallow)
	int op[], disallow;
{
	register char chr;

	op[2] = op[3] = 0;
	op[1] = 1;
	if((chr = operand[optr++]) == '#' ) {	/* immediate addressing */
		op[2] = eval();
		op[0] = IMMEDIATE;
		if(disallow & 0x02) reperr(3);
		return op[1] += !(oflags & BYTE); }

	if(chr == '[') {						/* Basic indirect */
		if((op[2] = eval()) & 0x01)
			reperr(4);
		expect(']');
		if(test_next('+')) {
			++optr;
			op[2] |= 1; }
		op[0] = INDIRECT;
		if(disallow & 0x04) reperr(3);
		return op[1]; }

	--optr;
	if(((op[2] = eval()) & 0x01) && !(oflags & 0x01))
		reperr(4);

	if(test_next('[')) {					/* Normal indexed */
		++op[1];
		op[3] = op[2];
		if((op[2] = eval()) & 0x01)
			reperr(4);
		if((op[3] >= 128) || (op[3] < -128)) {	/* Need long */
			++op[1];
			op[2] |= 1; }
		if(disallow & 0x08) reperr(3);
		op[0] = INDEXED;
		return op[1]; }

	if(op[2] & 0xFF00) {					/* Forced indexed */
		op[0] = INDEXED;
		op[3] = op[2];
		op[2] = 0;
		if((unsigned)op[3] < 0xFF80) {		/* Need long */
			op[2] = 1;
			++op[1]; }
		if(disallow & 0x08) reperr(3);
		return ++op[1]; }

	op[0] = REGISTER;
	if(disallow & 0x01) reperr(3);
	return op[1];
}

/*
 * Write out title
 */
write_title()
{
	int w;
	char *ptr;

	if(pcount > 1)
		putc('\f',lst_fp);
	lcount = 1;
	fprintf(lst_fp,"DUNFIELD 8096 ASSEMBLER: ");
	ptr = title;
	for(w = 35; *ptr; ++w)
		putc(*ptr++, lst_fp);
	while(w++ < pagew)
		putc(' ', lst_fp);
	fprintf(lst_fp,"PAGE: %u\n\n", pcount);
	++pcount;
}

/**************************************/
/* find next element in argument list */
/**************************************/
int	nxtelem() {
	register char chr;
	while((chr=operand[optr])&&(chr!= ' ')&&(chr!=9)) {
		++optr;
		if(chr=='\'') {
			while((chr=operand[optr])&&(chr!='\'')) ++optr;
			++optr; }
		if(chr==',') return(1); }
	return(0);
}

/*
 * Write an operand value into an instruction
 */
write_operand(op, pos)
	unsigned op[], pos;
{
/* printf("Wop @ %u: %u %u %02x %02x\n", pos, op[0], op[1], op[2], op[3]); */
	switch(op[0]) {
		case REGISTER :
		case INDIRECT :
			instruction[pos++] = op[2];
			break;
		case IMMEDIATE:
			instruction[pos++] = op[2];
			if(!(oflags & 0x01))
				instruction[pos++] = op[2] >> 8;
			break;
		case INDEXED :
			instruction[pos++] = op[2];
			instruction[pos++] = op[3];
			if(op[2] & 0x01)
				instruction[pos++] = op[3] >> 8; }
	return pos;
}

/*
 * Too many optimization passes - report failure
 */
optfail()
{
	fprintf(lst_fp,"** Line %u - Unable to resolve: %s\n", line, label);
	++ecnt;
}
