/*
 * AVR Cross Assembler
 *
 * ?COPY.TXT 1983-2005 Dave Dunfield
 * **See COPY.TXT**.
 */
#include <stdio.h>
#include <ctype.h>

#include "xasm.h"

/* AVR opcode table - format: text, type, length, opcode */
static char opcodes[] = {
	'C','L','C'+128,	1, 0x94, 0x88,
	'C','L','H'+128,	1, 0x94, 0xD8,
	'C','L','I'+128,	1, 0x94, 0xF8,
	'C','L','N'+128,	1, 0x94, 0xA8,
	'C','L','S'+128,	1, 0x94, 0xC8,
	'C','L','T'+128,	1, 0x94, 0xE8,
	'C','L','V'+128,	1, 0x94, 0xB8,
	'C','L','Z'+128,	1, 0x94, 0x98,
	'I','C','A','L','L'+128,1,0x95, 0x09,
	'I','J','M','P'+128,1, 0x94, 0x09,
	'L','P','M'+128,	1, 0x95, 0xC8,
	'E','L','P','M'+128,1, 0x95, 0xD8,
	'N','O','P'+128,	1, 0x00, 0x00,
	'R','E','T'+128,	1, 0x95, 0x08,
	'R','E','T','I'+128,1, 0x95, 0x18,
	'S','E','C'+128,	1, 0x94, 0x08,
	'S','E','H'+128,	1, 0x94, 0x58,
	'S','E','I'+128,	1, 0x94, 0x78,
	'S','E','N'+128,	1, 0x94, 0x28,
	'S','E','S'+128,	1, 0x94, 0x48,
	'S','E','T'+128,	1, 0x94, 0x68,
	'S','E','V'+128,	1, 0x94, 0x38,
	'S','E','Z'+128,	1, 0x94, 0x18,
	'S','L','E','E','P'+128,1,0x95,0x88,
	'W','D','R'+128,	1, 0x95, 0xA8,
	'A','S','R'+128,	2, 0x94, 0x05,
	'C','O','M'+128,	2, 0x94, 0x00,
	'D','E','C'+128,	2, 0x94, 0x0A,
	'I','N','C'+128,	2, 0x94, 0x03,
	'I','N','C'+128,	2, 0x94, 0x03,
	'L','S','R'+128,	2, 0x94, 0x06,
	'N','E','G'+128,	2, 0x94, 0x01,
	'P','O','P'+128,	2, 0x90, 0x0F,
	'P','U','S','H'+128,2, 0x92, 0x0F,
	'R','O','R'+128,	2, 0x94, 0x07,
	'S','W','A','P'+128,2, 0x94, 0x02,
	'A','D','C'+128,	3, 0x1C, 0x00,
	'A','D','D'+128,	3, 0x0C, 0x00,
	'A','N','D'+128,	3, 0x20, 0x00,
	'C','P'+128,		3, 0x14, 0x00,
	'C','P','C'+128,	3, 0x04, 0x00,
	'C','P','S','E'+128,3, 0x10, 0x00,
	'E','O','R'+128,	3, 0x24, 0x00,
	'M','O','V'+128,	3, 0x2C, 0x00,
	'M','U','L'+128,	3, 0x9C, 0x00,
	'O','R'+128,		3, 0x28, 0x00,
	'S','B','C'+128,	3, 0x08, 0x00,
	'S','U','B'+128,	3, 0x18, 0x00,
	'L','D','S'+128,	4, 0x90, 0x00,
	'S','T','S'+128,	5, 0x92, 0x00,
	'A','N','D','I'+128,6, 0x70, 0x00,
	'C','P','I'+128,	6, 0x30, 0x00,
	'L','D','I'+128,	6, 0xE0, 0x00,
	'O','R','I'+128,	6, 0x60, 0x00,
	'S','B','C','I'+128,6, 0x40, 0x00,
	'S','B','R'+128,	6, 0x60, 0x00,
	'S','U','B','I'+128,6, 0x50, 0x00,
	'A','D','I','W'+128,7, 0x96, 0x00,
	'S','B','I','W'+128,7, 0x97, 0x00,
	'B','C','L','R'+128,8, 0x94, 0x88,
	'B','S','E','T'+128,8, 0x94, 0x08,
	'B','L','D'+128,	9, 0xF8, 0x00,
	'B','S','T'+128,	9, 0xFA, 0x00,
	'S','B','R','C'+128,9, 0xFC, 0x00,
	'S','B','R','S'+128,9, 0xFE, 0x00,
	'B','R','B','C'+128,10,0xF4, 0x00,
	'B','R','B','S'+128,10,0xF0, 0x00,
	'B','R','C','C'+128,11,0xF4, 0x00,
	'B','R','C','S'+128,11,0xF0, 0x00,
	'B','R','E','Q'+128,11,0xF0, 0x01,
	'B','R','G','E'+128,11,0xF4, 0x04,
	'B','R','H','C'+128,11,0xF4, 0x05,
	'B','R','H','S'+128,11,0xF0, 0x05,
	'B','R','I','D'+128,11,0xF4, 0x07,
	'B','R','I','E'+128,11,0xF0, 0x07,
	'B','R','L','O'+128,11,0xF0, 0x00,
	'B','R','L','T'+128,11,0xF0, 0x04,
	'B','R','M','I'+128,11,0xF0, 0x02,
	'B','R','N','E'+128,11,0xF4, 0x01,
	'B','R','P','L'+128,11,0xF4, 0x02,
	'B','R','S','H'+128,11,0xF4, 0x00,
	'B','R','T','C'+128,11,0xF4, 0x06,
	'B','R','T','S'+128,11,0xF0, 0x06,
	'B','R','V','C'+128,11,0xF4, 0x03,
	'B','R','V','S'+128,11,0xF0, 0x03,
	'C','A','L','L'+128,12,0x94, 0x0E,
	'J','M','P'+128,	12,0x94, 0x0C,
	'R','C','A','L','L'+128,13,0xD0,0x00,
	'R','J','M','P'+128,13,0xC0, 0x00,
	'C','B','I'+128,	14,0x98, 0x00,
	'S','B','I'+128,	14,0x9A, 0x00,
	'S','B','I','C'+128,14,0x99, 0x00,
	'S','B','I','S'+128,14,0x9B, 0x00,
	'I','N'+128,		15,0xB0, 0x00,
	'O','U','T'+128,	16,0xB8, 0x00,
	'C','L','R'+128,	17,0x24, 0x00,
	'L','S','L'+128,	17,0x0C, 0x00,
	'R','O','L'+128,	17,0x1C, 0x00,
	'T','S','T'+128,	17,0x20, 0x00,
	'L','D'+128,		18,0x00, 0x00,
	'S','T'+128,		18,0x02, 0xFF,
	'L','D','D'+128,	19,0x80, 0x00,
	'S','T','D'+128,	19,0x82, 0x00,
	'C','B','R'+128,	20,0x70, 0x00,
	'S','E','R'+128,	21,0xE0, 0x00,
/* Directives */
	'E','Q','U'+128, 100, 0, 0,
	'O','R','G'+128, 101, 0, 0,
	'D','B'+128,     102, 0, 0,
	'D','W'+128,     103, -1,-1,
	'D','R','W'+128, 103, 0, 0,
	'D','S'+128,     104, 0, 0,
	'S','T','R'+128, 105, 0, 0,
	'S','T','R','H'+128,105,1,1,
	'S','T','R','Z'+128,105,2,2,
	'F','C','B'+128, 102, 0, 0,
	'F','D','B'+128, 103, -1,-1,
	'R','D','B'+128, 103, 0, 0,
	'R','M','B'+128, 104, 0, 0,
	'F','C','C'+128, 105, 0, 0,
	'F','C','C','H'+128,105,1,1,
	'F','C','C','Z'+128,105,2,2,
	'E','N','D'+128, 106, 0, 0,
	'C','P','A','G','E'+128,107,0,0,
	'P','A','G','E'+128,120,0,0,
	'T','I','T','L','E'+128,121,0,0,
	'S','P','A','C','E'+128,122,0,0,
	'L','I','S','T'+128,123,0,0,
	'N','O','L','I','S','T'+128,124,0,0,
	'L','I','N','E'+128,125,0,0,
	0 } ;

/* error message text table */
static char *etext[] = { "?",
	"Unknown instruction",
	"Out of range",
	"Invalid addressing mode",
	"Invalid register specification",
	"Undefined symbol",
	"Invalid expression syntax",
	"Invalid argument format",
	"Improperly delimited string",
	"Improper alignment" };

/* Symbol table & free pointer */
char symtab[SYMB_POOL], *symtop;

/* misc. global variables */
char opcode, opcode1, length, type, buffer[LINE_SIZE+1], label[SYMB_SIZE+1],
	operand[80], title[50], *optr;
unsigned char instruction[80], outrec[35];

unsigned addr, value, line, pcount, emsg, ecnt, lcount, ocnt, cpage,
	pagel=59, pagew=80;

char symf = 0, fulf = 0, intel = 0, quietf = 0, nlist = 0, casf = 0,
	rjmp = 0, rwrap = 0;

FILE *asm_fp, *hex_fp, *lst_fp;

unsigned char Lnoincdec[] = {
	0x90, 0x0C,		/* X */
	0x80, 0x08,		/* Y */
	0x80, 0x00 };	/* Z */
unsigned char Lpredec[] = {
	0x90, 0x0E,		/* -X */
	0x90, 0x0A,		/* -Y */
	0x90, 0x02 };	/* -Z */
unsigned char Lpostinc[] = {
	0x90, 0x0D,		/* X+ */
	0x90, 0x09,		/* Y+ */
	0x90, 0x01 };	/* Z+ */
/*
 * Define a symbol in the symbol table
 */
char *create_symbol(symbol, value)
	char *symbol;
	int value;
{
	register char *ptr, *ptr1;

	ptr = ptr1 = symtop;
	while(*symbol)
		*++ptr = *symbol++;
	*ptr1 = ptr - ptr1;
	*++ptr = value >> 8;
	*++ptr = value;
	if(ptr < (symtab+sizeof(symtab))) {
		symtop = ptr + 1;
		return ptr1; }
	return 0;
}

/*
 * Lookup a symbol in the symbol table
 */
char *lookup_symbol(symbol)
	char *symbol;
{
	register int l;
	register unsigned char *ptr, *ptr1;
	char *ptr2;

	ptr = symtab;
again:
	if((ptr2 = ptr) >= symtop) {
		value = 0x8888;
		return 0; }
	ptr1 = symbol;
	l = *ptr++ & SYMMASK;
	while(l--) {
		if(*ptr1++ != *ptr++) {
			ptr += l + 2;
			goto again; } }
	if(*ptr1) {
		ptr += 2;
		goto again; }
	value = *ptr++ << 8;
	value |= *ptr;
	return ptr2;
}

/*
 * Set the value for a symbol in the symbol table
 */
set_symbol(ptr, value)
	char *ptr;
	unsigned value;
{
	ptr += (*ptr & SYMMASK);
	*++ptr = value >> 8;
	*++ptr = value;
}

/*
 * Display the symbol table (sorted)
 */
dump_symbols()
{
	unsigned char *ptr, *hptr;
	unsigned int l, h, w;

	fprintf(lst_fp, "SYMBOL TABLE:\n\n");
	w = 0;
	for(;;) {
		for(hptr = symtab; hptr < symtop; hptr += (*hptr & SYMMASK) + 3)
			if(!(*hptr & SYMSORT))
				goto found;
		putc('\n', lst_fp);
		return;
found:	for(ptr = (*hptr & SYMMASK) + hptr + 3; ptr < symtop; ptr += (*ptr & SYMMASK) + 3) {
			if(*ptr & SYMSORT)
				continue;
			if(compsym(ptr, hptr) < 0)
				hptr = ptr; }
		*(ptr = hptr) |= SYMSORT;
		h = l = *ptr++ & SYMMASK;
		if((w + l + 7) >= pagew) {			/* Overrun page length */
			putc('\n', lst_fp);
			w = 0; }
		if(w) {								/* Not a start of line - separate */
			fprintf(lst_fp, "   ");
			w += 3; }
		while(l--)
			putc(*ptr++, lst_fp);
		w += (l = (h > 8) ? 24 : 8) + 5;	/* Calculate extended length */
		while(h++ < l)
			putc(' ', lst_fp);
		l = *ptr++ << 8;
		l |= *ptr++;
		fprintf(lst_fp, "-%04x", l); }
}

/*
 * Compare two symbol table entries
 */
compsym(sym1, sym2)
	char *sym1;
	char *sym2;
{
	int l, l1, l2;
	l1 = *sym1++ & SYMMASK;
	l2 = *sym2++ & SYMMASK;
	l = (l1 < l2) ? l1 : l2;
	do {
		if(*sym1 > *sym2)
			return 1;
		if(*sym1++ < *sym2++)
			return -1; }
	while(--l);
	if(l1 > l2)
		return 1;
	if(l1 < l2)
		return -1;
	return 0;
}

/*
 * Convert character to upper case if NOT case sensitive
 */
chupper(c)
	char c;
{
	return casf ? c : ((c >= 'a') && (c <= 'z')) ? c - ('a'-'A') : c;
}

/*
 * Open a filename with the appriopriate extension &
 * report an error if not possible
 */
FILE *open_file(filename, extension, options)
	char *filename, *extension, *options;
{
	char buffer[100], *ptr, *dot;
	FILE *fp;

	dot = 0;

	for(ptr = buffer; *ptr = *filename; ++ptr) {	/* Copy filename */
		if(*filename == '.')
			dot = filename;
		else if(*filename == '\\')
			dot = 0;
		++filename; }

	if(!dot) {									/* No extension supplied */
		*ptr++ = '.';
		do
			*ptr++ = *extension;
		while(*extension++); }
	else
		*dot = 0;

	if(!(fp = fopen(buffer, options))) {
		fprintf(stderr,"Unable to access: '%s'\n", buffer);
		exit(-1); }

	return fp;
}

/*
 * Main program
 */
main(argc, argv)
	unsigned argc;
	char *argv[];
{
	unsigned temp, i, daddr;
	char c, *ptr, *lfile, *cfile;
	int k;

	if(argc < 2)
		abort("\nUse: asmAVR <filename> [-acfiqstw c=file l=file p=length w=width]\n\n?COPY.TXT 1983-2005 Dave Dunfield\n**See COPY.TXT**.\n");

/* parse for command line options. */
	lfile = cfile = argv[1];
	for(i=2; i < argc; ++i) {
		if(*(ptr = argv[i]) == '-') {		/* Enable switch */
			while(*++ptr) switch(toupper(*ptr)) {
				case 'A' : rjmp = -1;		break;
				case 'C' : casf = -1;		break;
				case 'F' : fulf = -1;		break;
				case 'I' : intel = -1;		break;
				case 'Q' : quietf = -1;		break;
				case 'S' : symf = -1;		break;
				case 'T' : lfile = 0;		break;
				case 'W' : rwrap = -1;		break;
				default: goto badopt; }
			continue; }
		if(*++ptr == '=') switch(toupper(*(ptr++ - 1))) {
			case 'L' : lfile = ptr;			continue;
			case 'C' : cfile = ptr;			continue;
			case 'P' : pagel=atoi(ptr)-1;	continue;
			case 'W' : pagew=atoi(ptr);		continue; }
	badopt:
		fprintf(stderr,"Invalid option: %s\n", argv[i]);
		exit(-1); }

	asm_fp = open_file(argv[1], "ASM", "r");
	hex_fp = open_file(cfile, "HEX", "w");
	lst_fp = (lfile) ? open_file(lfile, "LST", "w") : stdout;
	strncpy(title,argv[1], sizeof(title)-1);

/* first pass - build symbol table */
	if(!quietf) fprintf(stderr,"First pass... ");
	symtop = symtab;
	addr = ocnt = ecnt = length = 0; line = 1;
	while(readline()) {
		if(optr) {
			if(label[0]) {
				if(ptr = lookup_symbol(label)) {
					fprintf(lst_fp,"** Line %u - Duplicate symbol: '%s'\n",line,label);
					++ecnt; }
				else {
					if(!(ptr = create_symbol(label, addr))) {
						xabort("symbol");
						fprintf(lst_fp,"** Line %u - Symbol table overflow\n", line);
						++ecnt;
						break; } } }
			locins(instruction);
			emsg = length = 0;
			optr=operand;
			switch(type) {
				case 1	:		/* No operands */
				case 2	:		/* Single destination register */
				case 3	:		/* Source & dest register */
				case 6	:		/* Dest R16-31 & 8-bit immediate */
				case 7	:		/* Dest Rpaid(24-30) & 6bit imm */
				case 8	:		/* Single flag address */
				case 9	:		/* Dest reg & bit number */
				case 10	:		/* Bit no & seven bit relative */
				case 11	:		/* Seven bit relative */
				case 13	:		/* 12 bit relative from PC */
				case 14	:		/* IO(0-31) & bit */
				case 15 :		/* Dest register & IO(0-31) */
				case 16	:		/* IO(0-31) & dest register */
				case 17 :		/* Duplicated register */
				case 18 :		/* Indexed */
				case 19 :		/* Index with offset */
				case 20 :		/* CBR */
				case 21 :		/* SER */
					length = 2;
					break;
				case 12	:		/* 22 bit immediate */
					if(rjmp && !nxtelem()) {
						length = 2;
						break; }
				case 4	:		/* Dest & mem address: Rd,addr */
				case 5	:		/* Mem address & Dest: addr,Rd */
					length = 4;
					break;
				case 100 :		/* EQU statement */
					set_symbol(ptr, eval(operand));
					break;
				case 101 :		/* ORG statement */
					addr = eval(operand);
					break;
				case 102 :		/* DB statement */
					length=1;
					while(nxtelem()) ++length;
					break;
				case 103 :		/* DW statement */
					length=2;
					while(nxtelem()) length += 2;
					break;
				case 104 :		/* DS statement */
					addr += eval(operand);
					break;
				case 105 :		/* STR statement */
					c=operand[0];
					for(i=1;(operand[i])&&(c!=operand[i]); ++i)
						++length;
					if(opcode == 2)
						++length;
					break;
				case 106 :		/* END statement */
					goto end1;
				case 125 :		/* LINE directive */
					line = eval(operand); }
			if(emsg) {
				fprintf(lst_fp,"** Line %u - %s\n",line,etext[emsg]);
				++ecnt; } }
		else type = length = 0;
		++line; /* += (type < 120); */
		addr += length; }

/* second pass - generate code */
end1:
	if(!quietf) fprintf(stderr,"Second Pass... ");
	addr = emsg = length = 0;
	lcount = 9999; pcount = line = 1;
	rewind(asm_fp);
	while(readline()) {
		daddr = addr;
		length = 0;
		if(optr) {
			if(!locins(instruction)) reperr(1); /* unknown opcode message */
			optr=operand;
			switch(type) {
				case 2	:		/* Single destination register */
					destreg(getreg());
				case 1	:		/* No operands */
					length=2;
					break;
				case 3	:		/* Source & Dest register */
					destreg(getreg());
					expect(',');
					srcreg(getreg());
					length = 2;
					break;
				case 4 :		/* Dest & mem address: Rd,addr */
					destreg(getreg());
					expect(',');
					instruction[2] = temp = eval(optr);
					instruction[3] = temp >> 8;
					length = 4;
					break;
				case 5	:		/* Mem address & Dest: addr,Rd */
					instruction[2] = temp = eval(operand);
					instruction[3] = temp >> 8;
					expect(',');
					destreg(getreg());
					length = 4;
					break;
				case 6	:		/* Dest R16-31 & 8-bit immediate */
				case 20	:		/* CBR */
				case 21 :		/* SER */
					if((temp = getreg()) < 0x10)
						reperr(4);
					destreg(temp & 0x0F);
					if(type == 21)
						temp = -1;
					else {
						expect(',');
						temp = eval(optr);
						if(type == 20)
							temp = ~temp; }
					opcode |= temp & 0x0F;
					opcode1 |= (temp >> 4) & 0x0F;
					length = 2;
					break;
				case 7	:		/* Dest Rpaid(24-30) & 6bit imm */
					switch(chupper(*optr)) {
						default: i = getreg(); goto noinc;
						case 'W' : i = 24;	break;
						case 'X' : i = 26;	break;
						case 'Y' : i = 28;	break;
						case 'Z' : i = 30;	}
					if(!isterm(*++optr))
						reperr(4);
				noinc:
					if(i & 0x01) reperr(4);
					if(i < 24)	reperr(4);
					opcode |= (i-24) << 3;
					expect(',');
					if((temp = eval(optr)) > 63)
						reperr(2);
					opcode |= (temp & 0x0F);
					opcode |= (temp << 2) & 0xC0;
					length = 2;
					break;
				case 8	:		/* Single flag address */
					opcode |= getbit(0) << 4;
					length = 2;
					break;
				case 9	:		/* Dest reg & bit number */
					destreg(getreg());
					expect(',');
					opcode |= getbit(0);
					length = 2;
					break;
				case 10	:		/* Bit no & seven bit relative */
					opcode |= getbit(0);
					expect(',');
				case 11 :		/* seven bit relative */
					k = (eval(optr) - daddr) - 2;
					if(k & 1)
						reperr(9);
					k /= 2;
					if((k > 63) || (k < -64))
						reperr(2);
					opcode |= k << 3;
					opcode1 |= (k >> 5) & 0x03;
					length = 2;
					break;
				case 12	:		/* 22 bit immediate */
					i = cpage;
					temp = eval(operand);
					if(*optr == ',') {
						++optr;
						if((i = temp) > 63)
							reperr(2);
						temp = eval(optr); }
					else if(rjmp) {
						opcode1 = (opcode == 0x0E) ? 0xD0 : 0xC0;
						goto dorel; }
					if(temp & 1)
						reperr(9);
					temp /= 2;
					opcode = opcode | ((i>>1) & 1) | ((i << 2) & 0xF0);
					opcode1 |= i >> 6;
					if(i & 1)
						temp |= 0x8000;
					instruction[2] = temp;
					instruction[3] = temp >> 8;
					length = 4;
					break;
				case 13	:		/* Relative 12 bit */
					temp = eval(optr);
				dorel:
					k = (temp - daddr) - 2;
					if(k & 1)
						reperr(9);
					k /= 2;
					if((k > 2047) || (k < -2048)) {
						if((temp > 8190) || !rwrap)
							reperr(2); }
					opcode = k;
					opcode1 |= (k >> 8) & 0x0F;
					length = 2;
					break;
				case 14	:		/* IO(0-31) & bit */
					if((temp = eval(operand)) > 31)
						reperr(2);
					expect(',');
					opcode = (temp << 3) | getbit(-1);
					length = 2;
					break;
				case 15	:		/* Dest register & IO(0-31) */
					destreg(getreg());
					expect(',');
					if((temp = eval(optr)) > 63)
						reperr(2);
					opcode |= temp & 0x0F;
					opcode1 |= (temp >> 3) & 0x06;
					length = 2;
					break;
				case 16	:		/* IO(0-31) & dest register */
					if((temp = eval(operand)) > 63)
						reperr(2);
					expect(',');
					destreg(getreg());
					opcode |= temp & 0x0F;
					opcode1 |= (temp >> 3) & 0x06;
					length = 2;
					break;
				case 17	:		/* Duplicated register */
					destreg(temp = getreg());
					srcreg(temp);
					length = 2;
					break;
				case 18 :		/* Indexed Load */
					if(!opcode) {
						temp = getreg();
						expect(','); }
					ptr = 0;
					if((c = *optr) == '-') {
						++optr;
						ptr = Lpredec; }
					i = (chupper(*optr++) - 'X') * 2;
					if(i > 4)
						reperr(4);
					if(*optr == '+') {
						++optr;
						if(ptr) reperr(3);
						ptr = Lpostinc; }
					if(!ptr) ptr = Lnoincdec;
					if(opcode) {
						expect(',');
						temp = getreg(); }
					if((*optr) && !isspace(*optr))
						reperr(7);
					opcode1 |= ptr[i];
					opcode = ptr[i+1];
					destreg(temp);
					length = 2;
					break;
				case 19	:		/* Indexed with offset */
					if(!(opcode1 & 0x02)) {
						temp = getreg();
						expect(','); }
					switch(chupper(*optr++)) {
						case 'Y' : opcode = 0x08; break;
						case 'Z' : opcode = 0x00; break;
						default: reperr(4); }
					expect('+');
					i = eval(optr);
					if(i & 0xFFC0) reperr(2);
					if(opcode1 & 0x02) {
						expect(',');
						temp = getreg(); }
					destreg(temp);
					opcode |= i & 7;
					opcode1 |= (i >> 1) & 0x0C;
					opcode1 |= i & 0x20;
					length = 2;
					break;
				case 100 :		/* EQU statement */
					daddr = eval(operand);	/* for error messages */
					break;
				case 101 :		/* ORG statement */
					if(ocnt) wrrec();
					daddr=addr=eval(operand);
					break ;
				case 102 :		/* DB statement */
					optr=operand;
					do	instruction[length++]=eval(optr);
					while(*optr++ == ',');
					goto fixins;
				case 103 :		/* DW statement */
					optr=operand;
					do {
						temp = eval(optr);
						if(opcode) {
							instruction[length++]=temp;
							instruction[length++]=temp >> 8; }
						else {
							instruction[length++]=temp >> 8;
							instruction[length++]=temp; } }
					while(*optr++ == ',');
					goto fixins;
				case 104 :		/* DS statement */
					if(ocnt) wrrec();
					addr += eval(operand);
					break;
				case 105 :		/* STR statement */
					c=operand[0];
					for(i=1;((operand[i])&&(c!=operand[i]));++i)
						instruction[length++] = operand[i];
					if(!operand[i]) reperr(8);
					if(opcode == 1)			/* STRH */
						instruction[length-1] |= 0x80;
					else if(opcode == 2)	/* STRZ */
						instruction[length++] = 0;
					goto fixins;
				case 106 :		/* END statement */
					goto end2;
				case 107 :		/* CPAGE statement */
					cpage = eval(operand);
					break;
				case 120 :		/* PAGE statement */
					lcount=9999;
					break ;
				case 121 :		/* TITLE statement */
					strncpy(title, operand, sizeof(title)-1);
					break;
				case 122 :		/* SPACE directive */
					fprintf(lst_fp,"\n");
					++lcount;
					break ;
				case 123 :		/* LIST directive */
					if(nlist) --nlist ;
					break;
				case 124 :		/* NOLIST directive */
					++nlist;
					break;
				case 125 :		/* LINE directive */
					line = eval(operand); }
	instruction[0]=opcode;
	instruction[1]=opcode1;
	if((daddr & 1) && (type < 100))
		reperr(9);
fixins:	; }
	else type = length = 0;

/* generate formatted output listing */
	if(((type<120) && fulf && !nlist) || emsg) {
		if(++lcount >= pagel)
			write_title();
		fprintf(lst_fp,"%04x ",daddr);
		for (i=0 ; i < 6; ++i) {
			if(i<length)
				fprintf(lst_fp," %02x",instruction[i]);
			else
				fprintf(lst_fp,"   "); }
		fprintf(lst_fp," %c%5u  %s\n", (length <= 6) ? ' ': '+', line,buffer); }
	if(emsg) {
		fprintf(lst_fp,"  ** ERROR ** - %u - %s\n",emsg,etext[emsg]);
		++ecnt; ++lcount;
		emsg=0; }
	++line; /* += (type < 120); */
/* write output to code file */
	for(i=0; i<length; ++i) {
		if(!ocnt) {						/* starting new record */
			outrec[ocnt++]=addr>>8;
			outrec[ocnt++]=addr; }
		++addr;
		outrec[ocnt++]=instruction[i];
		if(ocnt>33) wrrec(); } }	/* record is full, write it out. */

end2:
	if(ocnt) wrrec();				/* write last partial record */
	if(intel) fprintf(hex_fp,":00000001FF\n");
	else	  fprintf(hex_fp,"S9030000FC\n");
	if(ecnt) fprintf(lst_fp,"\n %u error(s) occurred in this assembly.\n",ecnt);
	if(!quietf) fprintf(stderr,"%u error(s).\n",ecnt);

/* display symbol table */
	if(symf) {
		write_title();
		dump_symbols(); }

/* close files */
	fclose(asm_fp);
	fclose(hex_fp);
	fclose(lst_fp);

	return ecnt ? -2 : 0;
}

/* Get a register identifier */
int getreg()
{
	unsigned r;
	if(chupper(*optr) != 'R') {
		reperr(4);
		return 0; }
	if(!isdigit(*++optr)) {
		reperr(4);
		return 0; }
	r = 0;
	while(isdigit(*optr))
		r = (r * 10) + (*optr++ - '0');
	if(r > 31) {
		reperr(4);
		return 0; }
	if(!isterm(*optr))
		reperr(4);
	return r;
}

/* Insert register into opcode as destination */
destreg(unsigned r)
{
	opcode  |= (r << 4) & 0xF0;
	opcode1 |= (r >> 4) & 0x01;
}

/* Insert a register into opcode as source */
srcreg(unsigned r)
{
	opcode  |= r & 0x0F;
	opcode1 |= (r >> 3) & 0x02;
}

/* Get a bit number (0-7) */
getbit(char flags)
{
	unsigned b;
	static char bits[] = { "CZNVSHTI" };
	char c;

	if(!flags) {		/* Allow CPU flags */
		c = chupper(*optr);
		if(isterm(optr[1])) {
			for(b=0; b < 8; ++b)
				if(bits[b] == c) {
					++optr;
					return b; } } }

	if((b = eval(optr)) > 7)
		reperr(2);

	return b;
}

/* Expect a specific character */
expect(char *c)
{
	if(*optr == c) {
		++optr;
		return; }
	reperr(7);
}

/***********************************************/
/* lookup instruction in the instruction table */
/***********************************************/
int locins(inst)
	char inst[];
{
	register char *ptr, *ptr1;

	ptr = opcodes;
	do {
		ptr1 = inst;
		while(*ptr == *ptr1) {
			++ptr;
			++ptr1; }
		if(((*ptr & 127) == *ptr1) && !*(ptr1+1)) {
			type = *++ptr;
			opcode1 = *++ptr;
			opcode = *++ptr;
			return 1; }
		while(*ptr++ >= 0); }
	while(*(ptr += 3));

	type = 119;
	return(inst[length = 0] == 0);
}

/*************************************************/
/* read a line from input file, and break it up. */
/*************************************************/
int readline()
{
	register int i;
	register char *ptr;

	if(!fgets(ptr = buffer, LINE_SIZE, asm_fp))
		return 0;

	i = 0;
	while(issymbol(*ptr)) {
		label[i] = chupper(*ptr++);
		if(i < SYMB_SIZE)
			++i; }
	label[i]=0;
	if(*ptr == ':')
		++ptr;
	while(isspace(*ptr))
		++ptr;
	if(((*ptr != '*') && (*ptr != ';')) || i) {
		i = 0;
		while(*ptr && !isspace(*ptr))
			instruction[i++] = toupper(*ptr++);
		instruction[i]=0;
		while(isspace(*ptr))
			++ptr;
		i = 0;
		while(*ptr && (i < 79))
			operand[i++] = *ptr++;
		(optr = operand)[i] = 0;
		return 1; }

	optr=0;
	return 1;
}

/*********************************************/
/* report an error, first error on line only */
/*********************************************/
reperr(en)
	int en;
{
	if (!emsg) emsg=en;
}

/**************************/
/* evaluate an expression */
/**************************/
int eval(cstr)
	char *cstr;
{
	register char c;
	unsigned result;

	optr=cstr;
	result=getval();
	while(!isterm(c=*optr)) {
		++optr;
		switch(c) {
			case '+' : result += getval();	break;
			case '-' : result -= getval();	break;
			case '*' : result *= getval();	break;
			case '/' : result /= getval();	break;
			case '\\': result %= getval();	break;
			case '&' : result &= getval();	break;
			case '|' : result |= getval();	break;
			case '^' : result ^= getval();	break;
			case '<' : result <<= getval();	break;
			case '>' : result >>= getval();	break;
			default: reperr(6); } }
	return result;
}

/***********************************************************/
/* get a value element from the expression being evaluated */
/***********************************************************/
int getval()
{
	unsigned i, b, val;
	char c, array[25], *ptr;

	switch(c = *optr++) {
		case '-' :
			return 0 - getval();
		case '~' :
			return ~getval();
		case '=' :
			i=getval();
			return (i<<8)+(i>>8);
		case '(' :
			val = eval(optr);
			if(*optr == ')')
				++optr;
			else
				reperr(6);
			return val; }
	--optr;

	val=b=i=0;
	switch(c) {
		case '$' :	b = 16;	goto getnum;
		case '@' :	b = 8;	goto getnum;
		case '%' :	b = 2;	goto getnum; }
	if(isdigit(c)) {
		array[i++] = c;
	getnum:
		while(isalnum(c = toupper(*++optr)))
			array[i++] = (c < 'A') ? c : c - 7;
		if((b == 16) && !i)
			val = addr;
		else {
			if(!b) {
				b = 10;
				switch(array[i-1]) {
					case 'H'-7 :	b = 16;	goto deci;
					case 'T'-7 :
					case 'D'-7 :	b = 10;	goto deci;
					case 'Q'-7 :
					case 'O'-7 :	b = 8;	goto deci;
					case 'B'-7 :	b = 2;
					deci:	--i;	} }
			if(!i)
				reperr(6);
			for(ptr = array; i; --i) {
				if((c = *ptr++ - '0') >= b)
					reperr(6);
				val = (val * b) + c; } } }
	else if(c==39) {					/* single quote. */
		++optr;
		while(((c=*optr++) != 39)&&c)
			val=(val << 8)+c;
		if(!c) reperr(8); }
	else if(c=='*') {					/* current program counter */
		++optr;
		val=addr; }
	else {								/* must be a label */
		i = 0;
		while(issymbol(c = chupper(*optr))) {
			++optr;
			label[i]=c;
			if(i < SYMB_SIZE)
				++i; }
		label[i]=0;
		if(!lookup_symbol(label)) reperr(5);
		val=value; }

	return val;
}

/***************************************/
/* test for valid terminator character */
/***************************************/
int isterm(c)
	char c;
{
	switch(c) {
		case 0 :
		case ' ' :
		case '\t':
		case ',' :
		case ')' :
			return 1; }
	return 0;
}

/*************************************/
/* Test for a valid symbol character */
/*************************************/
int issymbol(c)
	char c;
{
	switch(c) {			/* Special allowed characters */
		case '_' :
		case '?' :
		case '!' :
		case '.' :
			return 1; }

	return isalnum(c);
}


/*********************************************/
/* skip to next element in the argument list */
/*********************************************/
int nxtelem()
{
	register char c;
	while((c=*optr)&&(c!=' ')&&(c!=9)) {
		++optr;
		if(c==39) {
			while((c=*optr)&&(c!=39)) ++optr;
			++optr; }
		if(c==',')
			return 1; }
		return 0;
}

/*******************************/
/* write record to output file */
/*******************************/
wrrec()
{
	register unsigned i, chk, c;

	xhex(ocnt-2);
	if(intel) {					/* intel hex format */
		chk = outrec[0] + outrec[1] + ocnt - 2;
		fprintf(hex_fp,":%02x%02x%02x00", ocnt-2, outrec[0], outrec[1]);
		for(i=2; i<ocnt; ++i) {
			fprintf(hex_fp,"%02x", c = outrec[i]);
			chk += c; }
		fprintf(hex_fp,"%02x\n", 255 & (0-chk)); }
	else {						/* motorola hex format */
		chk = ocnt + 1;
		fprintf(hex_fp,"S1%02x", ocnt + 1);
		for(i=0; i<ocnt; ++i) {
			fprintf(hex_fp,"%02x", c = outrec[i]);
			chk += c; }
		fprintf(hex_fp,"%02x\n", 255 & ~chk); }
	ocnt = 0;
}

/*
 * Write out title
 */
write_title()
{
	int w;
	char *ptr;

	if(pcount > 1)
		putc('\f',lst_fp);
	lcount = 1;
	fprintf(lst_fp,"DUNFIELD AVR ASSEMBLER: ");
	ptr = title;
	for(w = 34; w < pagew; ++w)
		putc(*ptr ? *ptr++ : ' ', lst_fp);
	fprintf(lst_fp,"PAGE: %u\n\n", pcount);
	++pcount;
}
