/*
 * Assembly Translator - Main module
 *
 * ?COPY.TXT 2000-2005 Dave Dunfield
 * **See COPY.TXT**.
 *
 * DDS Micro-C/PC compile command: cc AT -pofm
 */
#include <stdio.h>

/* Variable size limits */
#define	LABEL_MAX	64			// Maximum width of label
#define	INST_MAX	30			// Maximum width of instruction
#define	OPER_MAX	100			// Maximum width of operand
#define	COMMENT_MAX	150			// Maximum width of comment field
#define	LINE_MAX	250			// Maximum width of input line
#define	VC_MAX		5			// Maximum number of variable strings
#define	VL_MAX		64			// Maximum length of variable strings
#define	OCLASS_MAX	50			// Maximum size of operand class

/* Operand clasification bytes */
#define VALUE	0xF0			// A value at position
#define	ERROR	0xF5			// Error indicator

/*
 * Command line options
 */
static unsigned char
	Debug = 0,					// Debugging messages
	Csource = 0,				// Source as comments in output
	Rcomment = 0,				// Remove comments
	Cmulti = 1,					// Multiple comments
	Scomment = ';',				// Source comment character
	Ocomment = ';',				// Output comment character
	Ldelim = 0,					// Label delimiter
	Icolumn = 8,				// Instruction column
	Ocolumn = 16,				// Operand Column
	Ccolumn = 40,				// Comment Column
	Tab_width = 8;				// Width of TAB's

/*
 * Global variables
 */
static FILE
	*ifp,						// Input file pounter
	*ofp,						// Output file pointer
	*efp;						// Error file pointer
static unsigned
	L, I, O,					// Lable, Inst, Operand lengths
	sline_number = 0,			// Current line number being read
	oline_number = 0,			// Current line number being written
	var_count,					// Count of variable arguments
	out_l;						// Output length
static unsigned char
	comment[COMMENT_MAX+1],		// Comment + Input buffer
	label[LABEL_MAX+1],			// Label on this line
	instruction[INST_MAX+1],	// Instruction on this line
	operand[OPER_MAX+1],		// Operand on this line
	opclass[OCLASS_MAX+1],		// Operand classification
	var_list[VC_MAX][VL_MAX+1],	// Maximum number of variables
	*input_ptr;					// General input parsing pointer

/*
 * External variables from the table module
 */
extern unsigned
	Slookup[],					/* Translation lookup table */
	Otranslate[];				/* Offsets into translate data */
extern char
	*Skeywords[],				/* Source keyword names */
	*Soperands[],				/* Source operand classifications */
	*Sinstructions[],			/* Source instruction names */
	*Okeywords[],				/* Output keyword names */
	*Oinstructions[],			/* Output instruction names */
	*Terrors[],					/* Translation error messages */
	Odata[];					/* Output translation data */
extern char
	*translate();				/* Expression translator */

/*
 * Test for an allowed symbol
 */
static int tstsymbol(c)
	char c;
{
	switch(c) {			/* Special allowed characters */
		case '_' :
		case '?' :
		case '!' :
			return 1; }

	return isalnum(c);
}

/*
 * Skip a single element value during classification
 */
static void skip_expression();
static void skip_element()
{
	char c;

top:
	switch(c = *input_ptr++) {
		case '-' :				/* Negated value */
		case '~' :				/* Complemented value */
		case '=' :				/* Swap high and low bytes */
			goto top;
		case '(' :				/* Nested expression */
			skip_expression();
			if(*input_ptr == ')')
				++input_ptr;
			return; }

	switch(c) {
		case '$' :
		case '@' :
		case '%' :
			goto skip_num;
		case '\'' :
			while(((c = *input_ptr++) != '\'') && c);
		case '*' :
			return; }
	if(isdigit(c)) {
	skip_num:
		while(isxdigit(*input_ptr))
			++input_ptr;
		switch(toupper(*input_ptr)) {
			case 'H' :
			case 'O' :
			case 'Q' :
			case 'T' :
				++input_ptr; }
		return; }

	if(tstsymbol(c)) {
		while(tstsymbol(*input_ptr))
			++input_ptr;
		return; }

	return;
}

/*
 * Skip a single expression value during classification
 */
static void skip_expression()
{
do_expr:
	skip_element();
	switch(*input_ptr) {
		case '+' :
		case '-' :
		case '*' :
		case '/' :
		case '\\':
		case '&' :
		case '|' :
		case '^' :
		case '<' :
		case '>' :
			++input_ptr;
			goto do_expr; }
}

/*
 * Locate a register name
 */
static int match_keyword()
{
	char *ptr, *ptr1;
	unsigned i;

	ptr = input_ptr;
	ptr1 = Skeywords[i = 0];

	do {
		while(*ptr1) {
			if(toupper(*input_ptr++) != *ptr1++)
				goto skip; }
		if(!tstsymbol(*input_ptr)) {
			return i|0x80; }
	skip: input_ptr = ptr; }
	while(ptr1 = Skeywords[++i]);

	return 0;
}

/*
 * Classify an operand type:
 */
static void classify_operand()
{
	unsigned char c, *oper_ptr, *ptr, *ptr1;

	oper_ptr = opclass;
	var_count = 0;

	while(*input_ptr) {
		if(tstsymbol(*input_ptr)) {
			if(c = match_keyword()) {
				*oper_ptr++ = c;
				continue; }
		do_expr:
			ptr = input_ptr;
			skip_expression();
			ptr1 = var_list[var_count++];
			while((ptr < input_ptr) && *ptr)
				*ptr1++ = *ptr++;
			*ptr1 = 0;
			*oper_ptr++ = VALUE;
			continue; }
		switch(*input_ptr) {
			case '-' :		/* Unary negate */
				if(input_ptr[1] == '-')
					break;
				ptr = input_ptr++;
				if(match_keyword()) {
					input_ptr = ptr;
					break; }
				input_ptr = ptr;
			case '~' :		/* Unary compliment */
			case '=' :		/* Unary swap */
			case '$' :		/* Hex constant */
			case '@' :		/* Octal constant */
			case '%' :		/* Binary constant */
			case '*' :		/* Program counter */
			case '(' :		/* Sub expression */
			case '\'' :		/* Quoted constant */
			goto do_expr; }
		*oper_ptr++ = *input_ptr++; }

	*oper_ptr = 0;
}

/*
 * Formatted print to log device
 */
static register log_error(args)
	unsigned args;
{
	char buffer[100];

	_format_(nargs() * 2 + &args, buffer);
	fprintf(efp, "[%u/%u] ", sline_number, oline_number);
	fputs(buffer, efp);
	putc('\n', efp);
}

/*
 * Test for a space or tab occuring in the input stream
 */
static int tstspace()
{
	return (*input_ptr == ' ') || (*input_ptr == '\t');
}

/*
 * Skip ahead to next non-blank in the input stream
 */
static int skip_blanks()
{
	while(tstspace())
		++input_ptr;
	return *input_ptr;
}

/*
 * Read a line from the input stream
 */
static int readline()
{
	unsigned char line[LINE_MAX+1], omode, c;

	*label = *instruction = *operand = *comment = L = I = O = omode = 0;

	if(!fgets(input_ptr = line, LINE_MAX, ifp))
		return 0;

	++sline_number;

	if(Debug)
		printf("--- %u ---> %s\n", sline_number, line);

	if(Csource) {			/* Original source as comments */
		putc(Ocomment, ofp);
		fputs(line, ofp);
		putc('\n', ofp); }


	/* If line is a comment, return with nothing */
	if(*input_ptr == Scomment) {
		strcpy(comment, line+1);
		return -1; }

	/* If a label is present, read it */
	if((!tstspace()) && *input_ptr) {
		while((!tstspace()) && *input_ptr)
			label[L++] = *input_ptr++;
		if(label[L-1] == ':')
			--L;
		if(Ldelim)
			label[L++] = Ldelim;
		label[L] = 0; }

	/* Record instruction */
	if(skip_blanks() == Scomment) {
		strcpy(comment, input_ptr+1);
		return -1; }
	while((!tstspace()) && *input_ptr) {
		instruction[I++] = toupper(*input_ptr++);
		instruction[I] = 0; }

	/* Record operand, squeeze out blanks where possible */
	while(c = *input_ptr++) {
		if(omode) {
			operand[O++] = c;
			if(c == omode)
				omode = 0;
			continue; }
		if(c == Scomment) {		/* Input comment */
			operand[O] = 0;
			strcpy(comment, input_ptr);
			return -1; }
		switch(c) {
			case '\'' :			/* Single quote */
			case '"' :			/* Double quote */
				omode = c;
			default:			/* Unrecognized character */
				operand[O++] = c;
			case ' ' :			/* Space characters */
			case '\t' :
				continue; } }
	operand[O] = 0;
	if(omode)
		log_error("Unterminated (%c) string in operand", omode);
	return -1;
}

/*
 * Tab the output over to a specified position
 */
static void tab(unsigned p)
{
	unsigned l;

	/* If already over position, output a single space */
	if((l = out_l) >= p) {
		putc(' ', ofp);
		++out_l;
		return; }

	/* Generate tabs until we are as close as possible to position */
	if(Tab_width) {
	dotab:
		while(++l % Tab_width);
		if(l <= p) {
			putc('\t', ofp);
			out_l = l;
			goto dotab; } }

	/* Fill with spaces if necessary */
	while(out_l < p) {
		putc(' ', ofp);
		++out_l; }
}

/*
 * Write output formatted line(s)
 */
static void write_output(unsigned i)
{
	unsigned char *ptr, *p, c;
	unsigned elist[10], ecount;

	ecount = out_l = 0;
	if(*label) {
		fputs(label, ofp);
		if((out_l = strlen(label)) >= Icolumn) {
			out_l = 0;
			putc('\n', ofp); } }

	ptr = (i != -1) ? (Odata + Otranslate[i]) : operand;
do_inst:
	tab(Icolumn);
	if(i == -1)
		fputs(p = instruction, ofp);
	else
		fputs(p = Oinstructions[*ptr++ - 1], ofp);
	out_l += strlen(p);
	if((*ptr != '\n') && *ptr)
		tab(Ocolumn);
	while(c = *ptr++) {
		if(c == '\n') {
			if(*comment) {
				tab(Ccolumn);
				putc(Ocomment, ofp);
				fputs(comment, ofp);
				switch(Cmulti) {
					case 0 : *comment = 0;	break;
					case 1 :
						strcpy(comment, " \"\""); } }
			putc('\n', ofp);
			++oline_number;
			out_l = 0;
			goto do_inst; }
		if(c >= ERROR) {
			elist[ecount++] = c - ERROR;
			continue; }
		if(c >= VALUE) {
			c -= VALUE;
			fputs(p = translate(var_list[c], i, c), ofp);
			out_l += strlen(p);
			continue; }
		if(c >= 0x80) {
			fputs(p = Okeywords[c - 0x80], ofp);
			out_l += strlen(p);
			continue; }
		putc(c, ofp);
		++out_l; }
	if(*comment) {
		tab(Ccolumn);
		putc(Ocomment, ofp);
		fputs(comment, ofp); }
	putc('\n', ofp);
	for(i=0; i < ecount; ++i)
		log_error(Terrors[elist[i]]);
	++oline_number;
}

static unsigned get_help();
main(int argc, char *argv[])
{
	unsigned I, O, i;
	char *ptr;

	efp = ofp = stdout;

	I = get_help();
	while(i = peek(get_cs(), I++))
		putc(i, stderr);

	if(argc < 2) {
	help:
		while(i = peek(get_cs(), I++))
			putc(i, stderr);
		exit(-1); }

	ifp = fopen(argv[1], "rvq");

	for(i=2; i < argc; ++i) {
		if(*(ptr = argv[i]) == '-') {
			switch((toupper(*++ptr) << 8) | toupper(*++ptr)) {
			case ('D'<<8)|'B' :	Debug = -1;					continue;
			case ('S'<<8)|'C' :	Csource = -1;				continue;
			case ('R'<<8)|'C' : Rcomment = -1;				continue;
			case ('M'<<8)|'C' : Cmulti = atoi(ptr+1);		continue;
			case ('N'<<8)|'C' : Cmulti = 1;					continue;
			case ('F'<<8)|'C' : Cmulti = 2;					continue; } }
		else if(ptr[2] == '=') {
			switch((toupper(*ptr++) << 8) | toupper(*ptr++)) {
			case ('C'<<8)|'S' : Scomment = *(ptr+1);		continue;
			case ('C'<<8)|'O' : Ocomment = *(ptr+1);		continue;
			case ('L'<<8)|'D' : Ldelim = *(ptr+1);			continue;
			case ('I'<<8)|'C' : Icolumn = atoi(ptr+1);		continue;
			case ('O'<<8)|'C' : Ocolumn = atoi(ptr+1);		continue;
			case ('C'<<8)|'C' : Ccolumn = atoi(ptr+1);		continue;
			case ('T'<<8)|'W' : Tab_width = atoi(ptr+1);	continue;
			case ('O'<<8)|'F' : ofp = fopen(ptr+1, "wvq");	continue;
			case ('E'<<8)|'F' : efp = fopen(ptr+1, "wvq");	continue; } }
		log_error("Invalid option: %s", argv[i]);
		goto help; }

	while(readline()) {
		if(Debug) {
			printf("L'%s' I'%s' O'%s' ", label, instruction, operand);
			printf("C'%s'\n", comment); }

		if(Rcomment)
			*comment = 0;

		if(!*instruction) {		/* No instruction on line */
			if(*label) {		/* Label is here */
				fputs(label, ofp);
				if(*comment) {
					if((out_l = strlen(label)) >= Ccolumn) {
						putc('\n', ofp);
						out_l = 0; }
					tab(Ccolumn);
					putc(Ocomment, ofp);
					fputs(comment, ofp); }
				putc('\n', ofp);
				continue; }
			if(!Rcomment) {
				if(*comment) {
					putc(Ocomment, ofp);
					fputs(comment, ofp); }
				putc('\n', ofp); }
			continue; }
			
		/* Lookup the instruction type */
		for(I=0; ptr = Sinstructions[I]; ++I) {
			if(!strcmp(instruction, ptr))
				goto found_i; }
		log_error("Instruction (%s) not known", instruction);
		I = O = -1;
	found_i:

		if(I != -1) {
			/* Lookup the operand type */
			input_ptr = operand;
			classify_operand();
			for(O=0; ptr = Soperands[O]; ++O) {
				if(!strcmp(opclass, ptr))
					goto found_o; }
			log_error("Operand (%s) not classified", operand);
			O = -1; }
	found_o:

		if(Debug) {
			printf("Inst: %d, Oper", I);
			if(O != -1) {
				putc('"', stdout);
				input_ptr = opclass;
				while(*input_ptr) {
					if(*input_ptr & 0x80)
						printf("\\x%02x", *input_ptr++);
					else
						putc(*input_ptr++, stdout); }
				putc('"', stdout); }
			printf(": %d = 0x%04x\n", O, (I << 8) | O);

			for(i=0; i < var_count; ++i)
				printf("$%u: %s\n", i, var_list[i]); }

		if((I != -1) && (O != -1)) {
			O |= (I << 8);
			for(i=0; (I = Slookup[i]) != -1; ++i) {
				if(I == O) {
					if(Debug)
						printf("Translation: %u\n", i);
					goto found_c; } }
			log_error("Instruction/Operand combination not known."); }
		write_output(-1);
		continue;
	found_c:
		write_output(i); }

	fclose(ifp);
	fclose(ofp);
	fclose(efp);
}

static unsigned get_help() asm
{
	MOV		AX,OFFSET eh
} asm {
eh:	DB	0Ah,'AT - Assembly Translator',0Ah,0Ah
	DB	'?COPY.TXT 2000-2005 Dave Dunfield',0Ah,'All rights reserved.',0Ah,0
	DB	0Ah,'Use: at <source file> [options]',0Ah,0Ah
	DB	'opts:	-DB	- enable DeBug output',0Ah
	DB	'	-MC	- Multi-line translations have no Comments',0Ah
	DB	'	-MC1	- Multi-line translations have "" Comments (*)',0Ah
	DB	'	-MC2	- Multi-line translations have full Comments',0Ah
	DB	'	-RC	- Remove comments from output',0Ah
	DB	'	-SC	- include Source in output as Comments',0Ah
	DB	'	CS=char	- set Comment character in Source (;)',0Ah
	DB	'	CO=char	- set Comment character in Output (;)',0Ah
	DB	'	LD=char	- set Label Delimiter character in output (none)',0Ah
	DB	'	IC=n	- set Instruction Column in output (8)',0Ah
	DB	'	OC=n	- set Operand Column in output (16)',0Ah
	DB	'	CC=n	- set Comment Column in output (40)',0Ah
	DB	'	TW=n	- set Tab Width in output, 0=spaces (8)',0Ah
	DB	'	OF=file	- set Output File to write (stdout)',0Ah
	DB	'	EF=file	- set Error/log File to write (stdout)',0Ah,0
}
