/*
 * Assembly Translator Compiler
 *
 * ?COPY.TXT 2000-2005 Dave Dunfield
 * **See COPY.TXT**.
 *
 * DDS Micro-C/PC compile command: cc ATC -pof m=s
 */
#include <stdio.h>

#define MAX_SINST	200		// # source instruction names
#define	MAX_SREG	30		// # source keywords
#define	MAX_SOPER	200		// # source operand classes
#define	MAX_OINST	200		// # output instruction names
#define	MAX_OREG	30		// # output keywords
#define	MAX_OOPER	4000	// # output translations
#define	MAX_ERROR	10		// # translate errors
#define	POOL_SIZE	40000	// Size of literal storage pool

#define	VALUE	0xF0		// First operand value
#define	ERROR	0xF5		// First error value

static FILE
	*ifp,					// Input file pointer
	*ofp;					// Output file pointer
static unsigned
	I, O, R;				// Instruction, Operand and Register counts
	SI,						// Source instruction index
	SO,						// Source operand index
	OI,						// Output instruction index
	OO,						// Output operand index
	line_number = 0,		// Current file line number
	sicount = 0,			// Source instruction count
	srcount = 0,			// Source register count
	socount = 0,			// Source operand count
	oicount = 0,			// Output instruction count
	orcount = 0,			// Output register count
	oocount = 0,			// Output operand count
	slookup[MAX_OOPER];		// Source lookup table
static unsigned char
	*input_ptr,				// General input pointer
	instruction[50],		// Parsed instruction
	reg[50],				// Parsed register/keyword
	operand[100],			// Parsed operand
	*sinst[250],			// Source instruction table
	*sreg[MAX_SREG],		// Source register/keyword table
	*soper[MAX_SOPER],		// Source operand table
	*oinst[MAX_SINST],		// Output instruction table
	*oreg[MAX_OREG],		// Output register/keyword table
	*ooper[MAX_OOPER],		// Output operand table
	*terrors[MAX_ERROR],	// Translation error messages
	pool[POOL_SIZE],		// Literal storage pool
	*pptr = &pool;			// Pool free pointer

/*
 * Formatted print to log device
 */
static register error(args)
	unsigned args;
{
	char buffer[100];

	_format_(nargs() * 2 + &args, buffer);
	fprintf(stderr, "[%u]: ", line_number);
	fputs(buffer, stderr);
	putc('\n', stderr);
	if(ofp)
		fclose(ofp);
	exit(-1);
}

/*
 * Test to see of the next character in the input stream is a "symbol"
 */
static int tstsymbol()
{
	switch(*input_ptr) {		/* Special allowed characters */
		case '_' :
		case '?' :
		case '!' :
			return 1; }

	return isalpha(*input_ptr);
}

/*
 * Test to see if the next character in the input stream is a "space"
 */
static int tstspace()
{
	return (*input_ptr == ' ') || (*input_ptr == '\t');
}

/*
 * Skip ahead to the next non-space in the input stream
 */
static char skip_blanks()
{
	while(tstspace())
		++input_ptr;
	return *input_ptr;
}

/*
 * Add a string to the global literal pool
 */
static void add_pool(char *p)
{
	#ifdef DEBUG
		printf("Add pool: ");
	#endif
	do {
		#ifdef DEBUG
			if(isprint(*p))
				putc(*p, stdout);
			else
				printf("\\x%02x", *p);
		#endif
		*pptr++ = *p; }
	while(*p++);
	#ifdef DEBUG
		putc('\n', stdout);
	#endif
}

/*
 * Read a line from the source file and process it
 * - Parse instruction and operands and fill in tables
 */
static int read_line()
{
	char buffer[200];
	unsigned i, j;

	I = O = 0;
again:
	if(!fgets(input_ptr = buffer, sizeof(buffer)-1, ifp))
		return 0;
	++line_number;
	if(*buffer == ';')
		goto again;
	if(!skip_blanks())
		goto again;

	if(*input_ptr == '~') {
		if(!isdigit(i = *++input_ptr))
			error("Error number must be ~0-9");
		++input_ptr;
		skip_blanks();
		terrors[i-'0'] = pptr;
		add_pool(input_ptr);
		goto again; }

	#ifdef DEBUG
		printf("%u: %s\n", line_number, buffer);
	#endif

	/* Parse instruction */
	if(*input_ptr == '|')
		error("No source instruction");
	while((!tstspace()) && *input_ptr)
		instruction[I++] = toupper(*input_ptr++);
	instruction[I] = 0;

	/* Lookup instruction in pool */
	for(SI=0; SI < sicount; ++SI)
		if(!strcmp(sinst[SI], instruction))
			goto found_i;
	/* Not found - add to pool */
	#ifdef DEBUG
		printf("SI: ");
	#endif
	sinst[sicount++] = pptr;
	add_pool(instruction);
found_i:

	/* Parse operand */
	if(!skip_blanks())
		error("No source operand");
	while((!tstspace()) && (*input_ptr != '|') && *input_ptr) {
		if(tstsymbol()) {			/* Register input */
			R = 0;
			while(tstsymbol() || isdigit(*input_ptr))
				reg[R++] = toupper(*input_ptr++);
			reg[R] = 0;
			for(i=0; i < srcount; ++i)
				if(!strcmp(sreg[i], reg))
					goto found_r;
			sreg[srcount++] = pptr;
			add_pool(reg);
		found_r:
			operand[O++] = i | 0x80;
			continue; }
		if(*input_ptr == '$') {				/* Value input */
			++input_ptr;
			operand[O++] = VALUE;
			continue; }
		operand[O++] = *input_ptr++; }
	operand[O] = 0;
	for(SO = 0; SO < socount; ++SO)
		if(!strcmp(soper[SO], operand))
			goto found_o;
	soper[socount++] = pptr;
	add_pool(operand);
found_o:

/* Process the translation instructions */

	O = 0;
next_inst:
	if(skip_blanks() != '|')
		error("No translation instruction");
	++input_ptr;
	if((!skip_blanks()) || (*input_ptr == '|'))
		error("No source instruction");
	I = 0;
	while((!tstspace()) && (*input_ptr != '|') && *input_ptr)
		instruction[I++] = toupper(*input_ptr++);
	instruction[I] = 0;

	/* Lookup instruction in pool */
	for(OI=0; OI < oicount; ++OI)
		if(!strcmp(oinst[OI], instruction))
			goto found_oi;
	/* Not found - add to pool */
	oinst[oicount++] = pptr;
	#ifdef DEBUG
		printf("OI: ");
	#endif
	add_pool(instruction);
found_oi:
	operand[O++] = OI+1;

	/* Parse operand */
	skip_blanks();
	while((!tstspace()) && (*input_ptr != '|') && *input_ptr) {
		if(*input_ptr == '$') {		/* Optional parameter */
			if(!isdigit(*++input_ptr))
				error("Invalid '$' variable");
			operand[O++] = (*input_ptr++ - '0') | VALUE;
			continue; }
		if(*input_ptr == '~') {		/* Error indicator */
			if(!isdigit(i = *++input_ptr))
				error("Error number must be ~0-9");
			if(!terrors[i -= '0'])
				error("Error (%u) has not been set", i);
			++input_ptr;
			operand[O++] = i + ERROR;
			continue; }
		if(tstsymbol()) {			/* Register input */
			R = 0;
			while(tstsymbol() || isdigit(*input_ptr))
				reg[R++] = toupper(*input_ptr++);
			reg[R] = 0;
			for(i=0; i < orcount; ++i)
				if(!strcmp(oreg[i], reg))
					goto found_or;
			oreg[orcount++] = pptr;
			add_pool(reg);
		found_or:
			operand[O++] = i | 0x80;
			continue; }
		operand[O++] = *input_ptr++; }

	if(*input_ptr == '|') {
		operand[O++] = '\n';
		goto next_inst; }
	operand[O] = 0;

	/* Add this instruction to the pool */
	ooper[oocount] = pptr;
	add_pool(operand);

	j = (SI << 8) + SO;
	for(i=0; i < oocount; ++i) {
		if(slookup[i] == j)
			error("Duplicate instruction/operand definition (%u)", i); }
	slookup[oocount++] = j;

	return -1;
}

/*
 * Setup for a comment to be output - space over to aligned column
 * and output the opening sequence.
 */
static void begin_comment(unsigned l, unsigned column)
{
	do {
		putc(' ', ofp); }
	while(++l < column);
	fputs("/* ", ofp);
}

/*
 * Output the string as a comment
 */
static unsigned print_comment(char *s, char *cm[])
{
	unsigned char c;

next_inst:
	if(cm == oreg)
		fprintf(ofp, "%s ", oinst[*s++ - 1]);
	while(c = *s++) {
		if(c & 0x80) {
			if(c >= ERROR) {
				if(cm == oreg)
					fprintf(ofp, "~%u", c - ERROR);
				else
					putc('$', ofp);
				continue; }
			if(c >= VALUE) {
				if(cm == oreg)
					fprintf(ofp, "$%u", c - VALUE);
				else
					putc('$', ofp);
				continue; }
			fputs(cm[c & 0x7F], ofp);
			continue; }
		if(c == '\n') {
			putc('|', ofp);
			goto next_inst; }
		putc(c, ofp); }
}

/*
 * Print a string to the output file
 */
static unsigned print_string(unsigned char *s)
{
	unsigned l;
	unsigned char c;
	
	l = 2;
	putc('"', ofp);
	if(s) {
		while(c = *s++) {
			++l;
			if(isprint(c)) {
				putc(c, ofp);
				continue; }
			if(c == '\n') {
				fputs("\\n", ofp);
				++l;
				continue; }
			fprintf(ofp, "\\x%02x", c);
			l += 3; } }
	putc('"', ofp);
	return l;
}

/*
 * Dump the data tables to the output file as C definitions
 */
static void dump_data()
{
	int i, j;
	unsigned char *ptr;

	fputs("unsigned Slookup[] = {		/* Source translation table */\n", ofp);
	for(i=0; i < oocount; ++i) {
		j = fprintf(ofp, "\t0x%04x", slookup[i]);
		begin_comment(j-2, 10);
		fprintf(ofp, "%u */,\n", i); }
	fputs("\t-1 };\n", ofp);

	fputs("char *Skeywords[] = {		/* Source keywords */\n", ofp);
	for(i=0; i < srcount; ++i) {
		putc('\t', ofp);
		j = print_string(sreg[i]);
		begin_comment(j, 30);
		fprintf(ofp, " %u */,\n", i); }
	fputs("\t0 };\n", ofp);

	fputs("char *Soperands[] = {		/* Source operand table */\n", ofp);
	for(i=0; i < socount; ++i) {
		putc('\t', ofp);
		j = print_string(ptr = soper[i]);
		begin_comment(j, 30);
		fprintf(ofp, "%u:", i);
		print_comment(ptr, sreg);
		fputs(" */,\n", ofp); }
	fputs("\t0 };\n", ofp);

	fputs("char *Sinstructions[] = {	/* Source instruction names */\n", ofp);
	for(i=0; i < sicount; ++i) {
		putc('\t', ofp);
		j = print_string(sinst[i]);
		begin_comment(j, 30);
		fprintf(ofp, "%u */,\n", i); }
	fputs("\t0 };\n", ofp);

	fputs("char *Okeywords[] = {		/* Output keywords */", ofp);
	for(i=0; i < orcount; ++i) {
		if(i)
			putc(',', ofp);
		fputs("\n\t", ofp);
		j = print_string(oreg[i]);
		begin_comment(j, 30);
		fprintf(ofp, "%u */", i); }
	fputs(" };\n", ofp);

	fputs("char *Oinstructions[] = {	/* Output instruction names */", ofp);
	for(i=0; i < oicount; ++i) {
		if(i)
			putc(',', ofp);
		fputs("\n\t", ofp);
		j = print_string(oinst[i]);
		begin_comment(j, 30);
		fprintf(ofp, "%u */", i); }
	fputs(" };\n", ofp);


	fputs("unsigned char Odata[] = {	/* Output translation data */", ofp);
	for(i=j=0; i < oocount; ++i) {
		ptr = ooper[i];
		slookup[i] = j;
		do {
			if(j)
				putc(',', ofp);
			if(!(j & 0x0F))
				putc('\n', ofp);
			fprintf(ofp,"0x%02x", *ptr);
			++j; }
		while(*ptr++); }
	fputs(" };\n", ofp);

	fputs("unsigned Otranslate[] = {		/* Operand offsets into data */", ofp);
	for(i=0; i < oocount; ++i) {
		if(i)
			putc(',', ofp);
		j = fprintf(ofp, "\n\t%u", slookup[i]);
		begin_comment(j-2, 10);
		fprintf(ofp, "%u:", i);
		print_comment(ooper[i], oreg);
		fputs(" */", ofp); }
	fputs(" };\n", ofp);

	fputs("char *Terrors[] = {			/* Translation error messages */", ofp);
	for(i=0; i < MAX_ERROR; ++i) {
		if(i)
			putc(',', ofp);
		fputs("\n\t", ofp);
		j = print_string(terrors[i]);
		begin_comment(j, 48);
		fprintf(ofp, "%u */", i); }
	fputs(" };\n", ofp);
}

/*
 * Main program
 */
static unsigned get_help();
main(int argc, char *argv[])
{
	ofp = stdout;
	I = get_help();
	while(O = peek(get_cs(), I++))
		putc(O, stdout);
	if(argc < 2) {
		while(O=peek(get_cs(), I++))
			putc(O, stdout);
		exit(-1); }

	concat(operand, argv[1], ".ATC");
	ifp = fopen(operand, "rvq");
	concat(operand, argv[1], ".C");
	ofp = fopen(operand, "wvq");

	fprintf(ofp, "/*\n * This is translation data for AT - DO NOT COMPILE TO EXE\n");
	fprintf(ofp, " *\n * Compile to a module with DDS Micro-C/PC: CC %s -FM\n", argv[1]);
	fprintf(ofp, " * Then link with AT + ATT modules: LC -s AT ATT %s\n */\n", argv[1]);

	while(read_line());

	fclose(ifp);

	dump_data();

	fclose(ofp);

	printf("---- Statistics ----   Used Avail\n");
	printf("Source instructions : %5u %5u\n", sicount, MAX_SINST);
	printf("Source keywords     : %5u %5u\n", srcount, MAX_SREG);
	printf("Source operand types: %5u %5u\n", socount, MAX_SOPER);
	printf("Output instructions : %5u %5u\n", oicount, MAX_OINST);
	printf("Output keywords     : %5u %5u\n", orcount, MAX_OREG);
	printf("Output translations : %5u %5u\n", oocount, MAX_OOPER);
	printf("Memory usage        : %5u %5u\n", pptr - pool, POOL_SIZE);
}

static unsigned get_help() asm
{
	MOV		AX,OFFSET eh
} asm {
eh:	DB	0Ah,'ATC - Assembly Translator Compiler',0Ah,0Ah
	DB	'?COPY.TXT 2000-2005 Dave Dunfield',0Ah,'All rights reserved.',0Ah,0Ah,0
	DB	'Use: atc <filename>',0Ah,0
}
