/*
 * Rom (hex file) manipulation utility
 *
 * ?COPY.TXT 1983-2005 Dave Dunfield
 * **See COPY.TXT**.
 */
#include <stdio.h>

unsigned
	chktab[10][3],		// Checksums to apply
	nchk,				// number of checksums
	psize = 8192,		// Rom size
	base = -1,			// Base address (not set)
	image,				// Image segment
	reclen = 34,		// Record length
	maxff = 5,			// Max $FF before split
	loadadr = -1,		// Load address
	maxsize,			// Maximum ROM size
	lcount,				// Load count
	ftop;				// # files loaded
unsigned char
	*ptr = 0,			// General pointer
	*string,			// General parse pointer
	buffer[256],		// General buffer
	fname[64],			// Filename to write
	*flist[10];			// List of files
char
	verbose = 0,		// Extra output flag
	wfile = 0,			// Write inhibit
	outfmt = 0,			// Output format (0=In 1=Bin 2=Mot 3=Int)
	defval = -1,		// Fill value
	endrec = -1,		// Write end record
	trunc;				// Truncate input

unsigned char help[] = { "\n\
Use:	hexfmt	load_file... [options]\n\n\
opts:	-B	= Binary output\n\
	-E	= inhibit End record\n\
	-I	= Intel output\n\
	-M	= Motorola output\n\
	-Q	= Quiet\n\
	-T	= Truncate image\n\
	-W	= inhibit output Write\n\
	B=value	= set image Base address\n\
	C=s,e,d	= Compute checksum (up to 10)\n\
	D=value	= Default fill value\n\
	F=value	= maximum FF before split\n\
	L=value	= set Load address\n\
	R=value	= output Record size\n\
	S=value	= Size of rom\n\
	W=file	= specify output file to Write\n\n\
1983-2005 Dave Dunfield ** See COPY.TXT **.\n" };

main(argc, argv)
	int argc;
	char *argv[];
{
	unsigned chksum, locn, i;

	for(i=1; i < argc; ++i) {
		if(*(string = argv[i]) == '-') {		/* Enable switch */
			while(*++string) switch(toupper(*string)) {
				case 'B' : outfmt = 1;		break;		/* Binary out */
				case 'M' : outfmt = 2;		break;		/* Motorola out */
				case 'I' : outfmt = 3;		break;		/* Intel out */
				case 'Q' : verbose = -1;	break;		/* Quiet mode */
				case 'T' : trunc = -1;		break;		/* Truncate */
				case 'W' : wfile = -1;		break;		/* Write inhibit */
				case 'E' : endrec = 0;		break;		/* Delete END record */
				default:
				badopt:
					fprintf(stderr,"HEXFMT: Invalid option: %s\n", argv[i]);
					exit(-1); }
			continue; }
		if(*++string == '=') switch(toupper(*(string++ - 1))) {
			case 'B' : base = get_num(0);	continue;	/* Specify base */
			case 'C' :		/* specify checksum start,end,location */
				chktab[nchk][0] = get_num(',');
				chktab[nchk][1] = get_num(',');
				chktab[nchk][2] = get_num(0);
				++nchk;
				continue;
			case 'D' : defval = get_num(0);	continue;	/* Default value */
			case 'F' : maxff = get_num(0);	continue;	/* Maximum FF's */
			case 'L' : loadadr = get_num(0);continue;	/* Load address */
			case 'R' : reclen = get_num(0)+2;continue;	/* Record length */
			case 'S' : psize = get_num(0);	continue;	/* Rom size */
			case 'W' : set_file(string);	continue;	/* Output file */
			default: goto badopt; }
		flist[ftop++] = argv[i]; }

	if(!ftop)
		abort(help);
//		abort("\nUse: hexfmt <load_file> [-beimqtw b=base c=start,end,loc* d=value\n	f=maxFF l=load_addr r=recsiz s=size w=file]\n\n?COPY.TXT 1983-2005 Dave Dunfield\n**See COPY.TXT**.\n");

	if(!(image = alloc_seg(4096)))
		abort("Cannot allocate memory");

	for(locn = 0; locn < psize; ++locn)
		poke(image, locn, defval);

	if(!*fname)
		set_file(flist[0]);

	for(i=0; i <  ftop; ++i)
		read_file(flist[i], image);

	if(loadadr == -1)
		loadadr = base;

	if(!verbose) {
		fprintf(stdout,"\nBase address is %04x, Load address is %04x\n", base, loadadr);
		fprintf(stdout,"Total of %u bytes loaded, Image size is %u\n", lcount, maxsize); }

	for(i=0; i<nchk; ++i) {		/* perform checksums */
		chksum = 0;
		for(locn = chktab[i][0]; locn <= chktab[i][1]; ++locn)
			chksum += peek(image, locn);	// image[locn] & 255;
		locn = chktab[i][2];
		poke(image, locn, chksum >> 8);
		poke(image, ++locn, chksum);
		if(!verbose)
			fprintf(stdout,"Checksum from %04x to %04x at %04x is %04x\n",
				chktab[i][0], chktab[i][1], chktab[i][2], chksum); }

	if(!verbose) {
		chksum = 0;
		for(locn = 0; locn < psize; ++locn)
			chksum += peek(image, locn);	// image[locn] & 255;
		fprintf(stdout,"ROM checksum is %04x\n",chksum); }

	if(loadadr != -1)
		base = loadadr;		/* set up new address */

	if(!wfile)			/* re-write file to disk */
		write_file(fname, image, trunc ? maxsize : psize);
}

/*
 * Read file into memory image
 */
read_file(char *filename, unsigned seg)
{
	unsigned a, b, chksum, loadptr, count;
	FILE *fp;

	if(!(fp = fopen(filename,"r"))) {
		fprintf(stderr,"HEXFMT: Unable to access: %s\n", filename);
		exit(-1); }

	for(;;) {
		if(!fgets(ptr = buffer, 80, fp)) {
			fprintf(stderr,"HEXFMT: No end of file record.\n");
			goto quit; }
		again: switch(*ptr++) {
			case 'S' :	/* Motorola HEX format */
				if(!outfmt) outfmt = 2;
				if(*ptr == '9') goto quit;
				if(*ptr++ != '1') continue;
				lcount += count = (chksum = get_byte()) - 3;
				chksum += a = get_byte();
				chksum += b = get_byte();
				loadptr = (a << 8) + b;
				if(base == -1) base = loadptr;
				if((loadptr -= base) > psize)
					abort("HEXFMT: ROM boundaries exceeded!");
				while(count--) {
					chksum += a = get_byte();
					poke(seg, loadptr++, a); }
				if((255 & ~chksum) != get_byte())
					goto quitbad;
				break;
			case ':' :		/* Intel HEX format */
				if(!outfmt) outfmt = 3;
				if(!(count = get_byte())) goto quit;
				lcount += count;
				chksum = (a = get_byte()) + count;
				chksum += b = get_byte();
				loadptr = (a << 8) + b;
				if(base == -1) base = loadptr;
				if((loadptr -= base) > psize)
					abort("HEXFMT: ROM boundaries exceeded!");
				chksum += get_byte();
				while(count--) {
					chksum += a = get_byte();
					poke(seg, loadptr++, a); }
				if((255 & -chksum) != get_byte())
					goto quitbad;
				break;
			case ' ' :	/* Space */
			case '\t' :	/* Tab */
				goto again;
			case 0 :		/* Null line */
				continue;
			default:
				abort("HEXFMT: Invalid record format."); }
		if(loadptr > maxsize)
			maxsize = loadptr;
		if(loadptr > psize)
			abort("HEXFMT: ROM boundaries exceeded!"); }

quitbad:
	abort("HEXFMT: Bad record checksum.");
quit:
	fclose(fp);
}

/* Load a byte from the hex file */
get_byte()
{
	return (get_hex(*ptr++) << 4) + get_hex(*ptr++);
}

/* Test for HEX digit & get value*/
get_hex(c)
	int c;
{
	if((c >= '0') && (c <= '9'))
		return c-'0';
	if((c >= 'A') && (c <= 'F'))
		return c-'A'+10;
	if((c >= 'a') && (c <= 'f'))
		return c-'a'+10;
	abort("HEXFMT: Bad hex digit.");
}

/*
 * Write file from memory image
 */
write_file(name, seg, size)
	char *name;
	unsigned seg, size;
{
	unsigned cnt, locn, adr1;
	char fflag;
	FILE *fp;

	cnt = locn = fflag = 0;

	if(!(fp = fopen(name, (outfmt > 1) ? "w" : "wb"))) {
		fprintf(stderr,"HEXFMT: Unable to write: '%s'\n", name);
		exit(-1); }

	if(outfmt > 1) {
		while(locn < size) {
			if(!cnt) {
				adr1 = loadadr + locn;
				buffer[cnt++] = adr1 >> 8;
				buffer[cnt++] = adr1 & 255; }
			if((buffer[cnt++] = peek(seg, locn++)) == 255) {
				if(!fflag) {
					for(adr1 = locn; (peek(seg,adr1) == 255) && (adr1 < size); ++adr1);
					if((adr1 - locn) >= maxff) {	/* within range of MAXFF */
						write_record(fp, cnt-1);
						locn = adr1;
						cnt = 0; }
					else
						fflag = -1; } }
			else
				fflag = 0;
			if(cnt >= reclen) {
				write_record(fp, cnt);
				cnt = 0; } }
		write_record(fp, cnt);
		if(endrec) switch(outfmt) {
			case 2 :
				fprintf(fp,"S9030000FC\n");
				break;
			case 3 :
				fprintf(fp,":00000001FF\n"); } }

	else			/* pure binary output */
		while(locn < size)
			putc(peek(seg, locn++), fp);

	fclose(fp);
}

/* write mhx record to a file */
write_record(fp, count)
	FILE *fp;
	unsigned count;
{
	unsigned chk, i, chr;

	if(count > 2) {
		switch(outfmt) {
			case 2 :		/* Motorola */
				fprintf(fp,"S1%02x", chk = count + 1);
				for(i=0; i < count; ++i) {
					fprintf(fp,"%02x", chr = buffer[i]);
					chk += chr; }
				fprintf(fp,"%02x\n",255&~chk);
				break;
			case 3 :		/* Intel */
				chk = buffer[0] + buffer[1] + count - 2;
				fprintf(fp,":%02x%02x%02x00",count-2, buffer[0], buffer[1]);
				for(i=2; i < count; ++i) {
					fprintf(fp,"%02x",chr = buffer[i]);
					chk += chr; }
				fprintf(fp,"%02x\n",255 & (0-chk)); } }
}

/* get a decimal/hex/octal/binary number from passed string */
get_num(term)
	char term;
{
	unsigned value;
	char chr;

	value = 0;

	if(isdigit(*string))				/* decimal number */
		while(isdigit(chr=*string++))
			value = (value * 10) + chr - '0';
	else if('$'==(chr=*string++))		/* hexidecimal number */
		while((isdigit(chr=toupper(*string++))) || ((chr >= 'A') && (chr <= 'F')))
			value = (value << 4) + ((chr < 'A') ? (chr - '0') : (chr - '7'));
	else if(chr=='@')					/* octal number */
		while(('0'<=(chr=*string++)) && (chr < '8'))
			value = (value << 3) + (chr - '0');
	else if(chr=='%')					/* binary number */
		while(('0'==(chr=*string++)) || (chr=='1'))
			value = (value << 1) + (chr - '0') ;
	else chr=-1;
	if(chr != term) abort("HEXFMT: Invalid operand\n");
	return(value);
}

/*
 * Set filename to write
 */
set_file(ptr)
	char *ptr;
{
	char *ptr1;

	ptr1 = fname;
	do
		*ptr1++ = *ptr;
	while(*ptr++);
}
