/*
 * Convert assembler source for use with XASM
 *
 * ?COPY.TXT 1992-2005 Dave Dunfield
 * **See COPY.TXT**.
 */
#include <stdio.h>

char *indirs[] = { "R0", "R1", "DPTR", "A+DPTR", "A+PC", 0 };

char labels = -1, numbers = -1, comment = -1, X8051 = -1;

extern unsigned IOB_size;

char help[] = "\n\
Performs most of the work necessary to convert Intel style source\n\
into the Motorola style source used by the XASM assemblers.\n\n\
Use: int2xasm <infile> <outfile> [opts]\n\n\
opts:	-C	Inhibit conversion of comments (; -> *)\n\
	-N	Inhibit convertion of constant number (xxh -> $xx etc)\n\
	-L	Inhibit convertion of labels (removes ':'s)\n\
	-@	Inhibit 8051 @ to [] translation\n\n\
?COPY.TXT 1992-2005 Dave Dunfield\n**See COPY.TXT**.\n";

/*
 * Test for valid HEX character
 */
ishex(c)
	int c;
{
	return (c >= '0' && c <= '9')
		|| (c >= 'a' && c <= 'f')
		|| (c >= 'A' && c <= 'F');
}

/*
 * Compare strings with case insensitivity
 */
compare(str1, str2)
	char *str1, *str2;
{
	do
		if(toupper(*str1++) != *str2)
			return 0;
	while(*str2++);
	return -1;
}

main(argc, argv)
	int argc;
	char *argv[];
{
	int c, d, i;
	char buffer[100], *ptr;
	FILE *in, *out;

	if(argc < 3)
		abort(help);

	for(i=3; i < argc; ++i) {
		if(*(ptr = argv[i]) == '-') {		/* Enable switch */
			while(*++ptr) switch(toupper(*ptr)) {
				case 'C' : comment = 0;	break;
				case 'L' : labels = 0;	break;
				case 'N' : numbers = 0;	break;
				case '@' : X8051 = 0;	break;
				default: goto badopt; }
			continue; }
	badopt:
		fputs("Invalid option: ", stderr);
		abort(argv[i]); }

	IOB_size = 8192;			/* Use 8K buffers for speed */
	in = fopen(argv[1], "rvq");
	out = fopen(argv[2], "wvq");

	d = '\n';			/* At beginning of line */
	while((c = getc(in)) >= 0) {
		if(comment && (c == ';') && (d == '\n'))
			c = '*';
		if((c == '\'') || (c == '"')) {
			putc(c, out);
			do
				putc(d = getc(in), out);
			while((d != c) && (d != '\n') && (d > 0));
			continue; }
		if(labels && (d == '\n') && issymbol(c)) {
			do
				putc(c, out);
			while(issymbol(c = getc(in)) || isdigit(c));
			if((d = c) != ':')
				putc(c, out);
			continue; }
		if(numbers && isdigit(c) && !issymbol(d) && !isdigit(d)) {		/* Beginning of a number */
			buffer[i=0] = c;
			while(ishex(c = getc(in)))
				buffer[++i] = c;
			switch(toupper(c)) {
				case 'O' :		/* Octal number */
				case 'Q' :		/* Another for of octal */
					putc('@', out);
					goto wnum;
				case 'H' :		/* Hex constant */
					putc('$', out);
				wnum:
					fput(buffer, i+1, out);
					d = 0;
					continue;
				default:
					if((d = toupper(buffer[i])) == 'D')	/* Decimal */
						--i;
					if(d == 'B') {						/* Binary number */
						--i;
						putc('%', out); }
					fput(buffer, i+1, out);
					putc(d = c, out);
					continue; } }
		if((c == '@') && X8051) {				/* 8051 indirection */
			i = 0;
			while(!isterm(c = getc(in)))
				buffer[i++] = c;
			buffer[i] = 0;
			for(i=0; ptr = indirs[i]; ++i)
				if(compare(buffer, ptr)) {
					putc('[', out);
					fputs(buffer, out);
					putc(d = ']', out);
					break; }
			if(!ptr) {
				putc('@', out);
				fputs(buffer, out); } }
		putc(d = c, out); }

	fclose(in);
	fclose(out);
}

/*
 * Test for a valid terminator character
 */
isterm(c)
	int c;
{
	return (c <= ' ') || (c == ',') || (c == ';') || (c == ':');
}

/*
 * Indicate if character is a valid symbol
 */
issymbol(c)
	char c;
{
	return	((c >= 'A') && (c <= 'Z'))
		||	((c >= 'a') && (c <= 'z'))
		||	(c == '_');
}
