#include <stdio.h>

/* Global storage */
char buffer[500], *sptr;
char label[200], instruction[200], argument[200];

/* Command line option variables */
static char x8051 = 0, intel = 0;

/*
 * Internal register names for the 8051
 */
char *reg51[] = {
	"A", "B", "PSW", "SP", "DPH", "DPL",
	"P0", "P1", "P2", "P3", "IP", "IE",
	"TMOD", "TCON", "TH0", "TL0", "TH1", "TL1", "TH2", "TL2", "T2CON",
	"SCON", "SBUF", "PCON", "RCAP2H", "RCAP2L", "ACC",
	0 };

/*
 * Table of 8051 BIT names & addresses
 */
char *bit51[] = {
	"TCON", "TF1", "TR1", "TF0", "TR0", "IE1", "IT1", "IE0", "IT0",
	"SCON",	"SM0", "SM1", "SM2", "REN", "TB8", "RB8", "TI",  "RI",
	"IE",   "EA",  0,    "ET2",  "ES",  "ET1", "EX1", "ET0", "EX0",
	"IP",	0,     0,    "PT2",  "PS",  "PT1", "PX1", "PT0", "PX0",
	"PSW",  "CY",  "AC", "F0",   "RS1", "RS0", "OV",  0,     "P",
	0 };


char help_text[] = { "\n\
Performs most of the work required to convert PseudoSam\n\
format assembly sources into DDS XASM format sources:\n\n\
	H\'n  B\'n  OQ\'n	-> $n  %n  @n\n\
	<<  >>		-> <  >\n\
	.equ sym,value	-> sym EQU value\n\
	.org		-> ORG\n\
	.dw  .drw  .rs	-> FDB  RDB  RMB\n\
	.db		-> FCB for numbers, FCC for strings\n\
	.eject		-> PAGE\n\
	.page		-> ORG (*+$FF)&$100\n\
	.end		-> removed\n\
\nUse: sam2xasm <infile> [outfile] [opts]\n\n\
opts:	-5	Remove 8051 SFR definitions, Add BIT definitions\n\
	-I	Use DS/DB/DW/DRW/STR instead of RMB/FCB/FDB/RDB/FCC\n\
\n?COPY.TXT 1998-2005 Dave Dunfield\n**See COPY.TXT**.\n" };


/*
 * Skip blanks in the input line, and return the non-blank found
 */
int skip_blanks()
{
	while(isspace(*sptr))
		++sptr;
	return *sptr;
}

/*
 * Parse a blank-delimited string
 */
void parse(char *dest)
{
	while(*sptr && !isspace(*sptr))
		*dest++ = *sptr++;
	*dest = 0;
}

/*
 * Parse an argument. with conversions
 */
void parse_arg(char *dest)
{
	char c;
	while(c = *sptr) {
		switch(c) {
			case 'h' :		// HEX numbers
			case 'H' :
				if((sptr[1] == '\'') && isxdigit(sptr[2])) {
					*dest++ = '$';
					sptr += 2;
					continue; }
				break;
			case 'b' :		// BINARY numbers
			case 'B' :
				if((sptr[1] == '\'') && isxdigit(sptr[2])) {
					*dest++ = '%';
					sptr += 2;
					continue; }
				break;
			case 'o' :		// OCTAL numbers
			case 'O' :
			case 'q' :
			case 'Q' :
				if((sptr[1] == '\'') && isxdigit(sptr[2])) {
					*dest++ = '@';
					sptr += 2;
					continue; }
				break;
			case '>' :		// SHIFT RIGHT
				if(sptr[1] == '>')
					++sptr;
				break;
			case '<' :		// SHIFT LEFT
				if(sptr[1] == '<')
					++sptr;
				break;
			case ';' :		// COMMENT - end of argument
				*dest = 0;
				return;
			case '\'' :		// Quoted strings
			case '\"' :
				*dest++ = c;
				++sptr;
				while(*sptr) {
					*dest++ = *sptr++;
					if(*sptr == c)
						break; }
				continue;
			case ' ' :		// Remove blanks
			case '\t' :
				++sptr;
				continue; }
		*dest++ = *sptr++; }
	*dest = 0;
}

/*
 * Case insensitive match of strings
 */
int match(char *source, char *test)
{
	while(*source)
		if(toupper(*source++) != *test++)
			return 0;
	return !*test;
}

main(int argc, char *argv[])
{
	int i, j;
	FILE *fpi, *fpo;
	char symbol[100], *ptr;

	fpi = 0;
	fpo = stdout;
	for(i=1; i < argc; ++i) {
		ptr = argv[i];
		if(*ptr == '-') {
			switch(toupper(*++ptr)) {
				case '5' : x8051 = -1;	continue;
				case 'I' : intel = -1;	continue; }
			abort("Unknown option"); }
		if(fpo != stdout)
			abort("Too many filenames");
		if(fpi)
			fpo = fopen(argv[i], "wvq");
		else
			fpi = fopen(argv[i], "rvq"); }

	if(!fpi)
		abort(help_text);

	if(x8051) {
		fputs("; 8051 BIT definitions\n", fpo);
		for(j=0; bit51[j]; j += 9)
			for(i=0; i < 8; ++i)
				if(ptr = bit51[j+i+1])
					fprintf(fpo, "%s\tEQU\t%s.%u\n", ptr, bit51[j], 7-i); }

	while(fgets(sptr = buffer, sizeof(buffer), fpi)) {
		*label = *instruction = *argument = 0;

		if(skip_blanks() == ';') {
			fputs(buffer, fpo);
			putc('\n', fpo);
			continue; }

		parse(symbol);
		if(*(sptr - 1) == ':') {
			strcpy(label, symbol);
			if(skip_blanks() == ';') {
				fprintf(fpo, "%s\tEQU\t*\t%s", label, sptr);
				continue; }
			parse(symbol); }

		if(match(symbol, ".EQU")) {
			strcpy(instruction, "EQU");
			ptr = label;
			while(*sptr) {
				if(*sptr == ',') {
					++sptr;
					*ptr = 0;
					parse_arg(argument);
					break; }
				if(!isspace(*sptr))
					*ptr++ = *sptr;
				++sptr; }
			if(x8051) {
				for(i=0; reg51[i]; ++i)
					if(match(label, reg51[i]))
						goto skip; } }
		else if(match(symbol, ".ORG")) {
			strcpy(instruction, "ORG");
			parse_arg(argument); }
		else if(match(symbol, ".RS")) {
			strcpy(instruction, intel ? "DS" : "RMB");
			parse_arg(argument); }
		else if(match(symbol, ".END")) {
			skip: continue; }
		else if(match(symbol, ".DB")) {
			fprintf(fpo, "%s", label);
			parse_arg(ptr = buffer);
			while(*ptr) {
				if(*ptr == ',') {
					++ptr;
					continue; }
				if(*ptr == '\"') {	/* String */
					fputs(intel ? "\tSTR\t\"" : "\tFCC\t\"", fpo);
					++ptr;
					while(*ptr) {
						putc(*ptr, fpo);
						if(*ptr++ == '\"')
							break; }
					if(*sptr == ';') {
						putc('\t', fpo);
						fputs(sptr, fpo); }
					putc('\n', fpo);
					continue; }
				fputs(intel ? "\tDB\t" : "\tFCB\t", fpo);
				j = 0;
				while(*ptr) {
					if(*ptr == '\"')
						break;
					if(*ptr == ',') {
						++ptr;
						j = -1;
						continue; }
					if(j) {
						putc(',', fpo);
						j = 0; }
					putc(*ptr++, fpo); }
				if(*sptr == ';') {
					putc('\t', fpo);
					fputs(sptr, fpo); }
				putc('\n', fpo); }
			continue; }
		else if(match(symbol, ".DW")) {
			strcpy(instruction, intel ? "DW" : "FDB");
			parse_arg(argument); }
		else if(match(symbol, ".DRW")) {
			strcpy(instruction, intel ? "DRW" : "RDB");
			parse_arg(argument); }
		else if(match(symbol, ".EJECT"))
			strcpy(instruction, "PAGE");
		else if(match(symbol, ".PAGE")) {
			strcpy(instruction, "ORG");
			strcpy(argument, "(*+$FF)&$100"); }
		else {
			strcpy(instruction, symbol);
			parse_arg(argument); }

		fputs(label, fpo);
		if(*instruction || *argument || *sptr) {
			putc('\t', fpo);
			fputs(instruction, fpo);
			if(*argument || *sptr) {
				putc('\t', fpo);
				fputs(argument, fpo);
				if(skip_blanks()) {
					if((i = 24 - strlen(argument)) <= 8)
						putc(' ', fpo);
					else do 
						putc('\t', fpo);
					while((i -= 8) > 8);
					fputs(sptr, fpo); } } }
		putc('\n', fpo); }

	fclose(fpo);
	fclose(fpi);
}
