/*
 * DDS: Universial Cross assembler
 *
 * This assembler requires three files which define the instruction encoding
 * of the processor:
 *
 *   UASM.001  - All global data and tables
 *   UASM.002  - Special case handlers for PASS 1
 *   UASM.003  - Special case handlers for PASS 2
 *
 * The files are created with the Universal Assembler COMPiler (UCOMP) using
 * a "source" file defining the instruction encoding. For example, to create
 * The 68HC16 assembler, execute 'UCOMP HC16', and then compile this program.
 *
 * ?COPY.TXT 1983-2005 Dave Dunfield
 *  -- see COPY.TXT --.
 */
#include <stdio.h>
#include <ctype.h>
#include "xasm.h"

/* Operand types */
#define	NONE		0
#define	IMM8		1
#define	IMM16		2
#define	VAL8		3
#define	VAL16		4
#define	KEYWORD		5

/* Size flags */
#define	VWORD		0x08
#define	VBYTE		0x04
#define	IWORD		0x02
#define	IBYTE		0x01

/* Error messages */
#define	ERRbadins	1
#define	ERRsymbol	2
#define	ERRformat	3
#define	ERRinvadr	4
#define	ERRstring	5
#define	ERRbrack	6
#define	ERRrange	7
#define	ERRexpre	8

static char *etext[] = { "?",
	"Unknown instruction",
	"Undefined symbol",
	"Invalid argument format",
	"Invalid addressing mode",
	"Improperly delimited string",
	"Improperly nested brackets",
	"Address out of range",
	"Invalid expression syntax" } ;

/* Symbol table & free pointer */
char symtab[SYMB_POOL], *symtop;

/* Misc. global variables */
char buffer[LINE_SIZE+1], label[SYMB_SIZE+1], title[50], instruction[81],
	operand[101], outrec[35];
unsigned user[4];		/* User supplied variables */

char optf=3, symf=0, intel=0, fulf=0, quietf=0, nlist=0, casf=0;

unsigned addr, line=0, pcount=0, lcount=0, value, length, emsg, ecnt,
	ocnt=0, pagel = 59, pagew = 80, ovalue[4], itype;

char *optr, operand0, operand1;

FILE *asm_fp, *hex_fp, *lst_fp;

/* Default calculation for relative addresses */
#define	RELCOMP	addr + length + (l & 0x40) ? 2 : 1

#include "UASM.001"

/*
 * Define a symbol in the symbol table
 */
char *create_symbol(symbol, value)
	char *symbol;
	int value;
{
	register char *ptr, *ptr1;

	ptr = ptr1 = symtop;
	while(*symbol)
		*++ptr = *symbol++;
	*ptr1 = ptr - ptr1;
	*++ptr = value >> 8;
	*++ptr = value;
	if(ptr < (symtab+sizeof(symtab))) {
		symtop = ptr + 1;
		return ptr1; }
	return 0;
}

/*
 * Lookup a symbol in the symbol table
 */
char *lookup_symbol(symbol)
	char *symbol;
{
	register int l;
	register unsigned char *ptr, *ptr1;
	char *ptr2;

	ptr = symtab;
again:
	if((ptr2 = ptr) >= symtop) {
		value = 0x8888;
		return 0; }
	ptr1 = symbol;
	l = *ptr++ & SYMMASK;
	while(l--) {
		if(*ptr1++ != *ptr++) {
			ptr += l + 2;
			goto again; } }
	if(*ptr1) {
		ptr += 2;
		goto again; }
	value = *ptr++ << 8;
	value |= *ptr;
	return ptr2;
}

/*
 * Set the value for a symbol in the symbol table
 */
set_symbol(ptr, value)
	char *ptr;
	unsigned value;
{
	ptr += (*ptr & SYMMASK);
	*++ptr = value >> 8;
	*++ptr = value;
}

/*
 * Display the symbol table (sorted)
 */
dump_symbols()
{
	unsigned char *ptr, *hptr;
	unsigned int l, h, w;

	fprintf(lst_fp, "SYMBOL TABLE:\n\n");
	w = 0;
	for(;;) {
		for(hptr = symtab; hptr < symtop; hptr += (*hptr & SYMMASK) + 3)
			if(!(*hptr & SYMSORT))
				goto found;
		putc('\n', lst_fp);
		return;
found:	for(ptr = (*hptr & SYMMASK) + hptr + 3; ptr < symtop; ptr += (*ptr & SYMMASK) + 3) {
			if(*ptr & SYMSORT)
				continue;
			if(compsym(ptr, hptr) < 0)
				hptr = ptr; }
		*(ptr = hptr) |= SYMSORT;
		h = l = *ptr++ & SYMMASK;
		if((w + l + 7) >= pagew) {			/* Overrun page length */
			putc('\n', lst_fp);
			w = 0; }
		if(w) {								/* Not a start of line - separate */
			fprintf(lst_fp, "   ");
			w += 3; }
		while(l--)
			putc(*ptr++, lst_fp);
		w += (l = (h > 8) ? 24 : 8) + 5;	/* Calculate extended length */
		while(h++ < l)
			putc(' ', lst_fp);
		l = *ptr++ << 8;
		l |= *ptr++;
		fprintf(lst_fp, "-%04x", l); }
}

/*
 * Compare two symbol table entries
 */
compsym(sym1, sym2)
	char *sym1;
	char *sym2;
{
	int l, l1, l2;
	l1 = *sym1++ & SYMMASK;
	l2 = *sym2++ & SYMMASK;
	l = (l1 < l2) ? l1 : l2;
	do {
		if(*sym1 > *sym2)
			return 1;
		if(*sym1++ < *sym2++)
			return -1; }
	while(--l);
	if(l1 > l2)
		return 1;
	if(l1 < l2)
		return -1;
	return 0;
}

/*
 * Convert character to upper case if NOT case sensitive
 */
chupper(c)
	char c;
{
	return casf ? c : ((c >= 'a') && (c <= 'z')) ? c - ('a'-'A') : c;
}

/*
 * Open a filename with the appriopriate extension &
 * report an error if not possible
 */
FILE *open_file(filename, extension, options)
	char *filename, *extension, *options;
{
	char buffer[100], *ptr, *dot;
	FILE *fp;

	dot = 0;

	for(ptr = buffer; *ptr = *filename; ++ptr) {	/* Copy filename */
		if(*filename == '.')
			dot = filename;
		else if(*filename == '\\')
			dot = 0;
		++filename; }

	if(!dot) {									/* No extension supplied */
		*ptr++ = '.';
		do
			*ptr++ = *extension;
		while(*extension++); }
	else
		*dot = 0;

	if(!(fp = fopen(buffer, options))) {
		fprintf(stderr,"Unable to access: '%s'\n", buffer);
		exit(-1); }

	return fp;
}

/*
 * Main program
 */
main(argc, argv)
	unsigned argc;
	char *argv[];
{	unsigned i, j, temp, daddr;
	char chr, cflg, pflg, *ptr, *lfile, *cfile;
	cflg = pflg = 0;

	if(argc < 2)
		abort("\nUse: asm"#CPU_SHORT#" <filename> [-cfiqst c=file l=file o=n p=length w=width]\n\n?COPY.TXT 1983-2005 Dave Dunfield\n -- see COPY.TXT --.\n");

/* parse for command line options. */
	lfile = cfile = argv[1];
	for(i=2; i < argc; ++i) {
		if(*(ptr = argv[i]) == '-') {		/* Enable switch */
			while(*++ptr) switch(toupper(*ptr)) {
				case 'C' : casf = -1;		break;
				case 'F' : fulf = -1;		break;
				case 'I' : intel = -1;		break;
				case 'Q' : quietf = -1;		break;
				case 'S' : symf = -1;		break;
				case 'T' : lfile = 0;		break;
				default: goto badopt; }
			continue; }
		if(*++ptr == '=') switch(toupper(*(ptr++ - 1))) {
			case 'L' : lfile = ptr;			continue;
			case 'C' : cfile = ptr;			continue;
			case 'O' : optf=atoi(ptr);		continue;
			case 'P' : pagel=atoi(ptr)-1;	continue;
			case 'W' : pagew=atoi(ptr);		continue; }
	badopt:
		fprintf(stderr,"Invalid option: %s\n", argv[i]);
		exit(-1); }

	asm_fp = open_file(argv[1], "ASM", "r");
	hex_fp = open_file(cfile, "HEX", "w");
	lst_fp = (lfile) ? open_file(lfile, "LST", "w") : stdout;
	strncpy(title, argv[1], sizeof(title)-1);

/* first pass - build symbol table */
	symtop = symtab;
	do {
		addr = cflg = 0; line = 1;
		if(!quietf) if(pflg) fprintf(stderr,"Opt... ");
			else	fprintf(stderr,"First pass... ");
		while(readline()) {
			if(optr) {
				itype = lookup_instruction(instruction);
				if(label[0]) {				/* label exists */
					if(pflg) {				/* otimization pass */
						ptr = lookup_symbol(label);
						if(itype == 0x7F01) {	/* EQU statement */
							j = value;
							if((temp = get_value()) != j) {
								if(pflg >= optf) optfail();
								set_symbol(ptr, temp);
								cflg=1; } }
						else if(value != addr) {
							if(pflg >= optf) optfail();
							set_symbol(ptr, addr);
							cflg=1; } }
					else {							/* original pass-1 */
						if(lookup_symbol(label)) {	/* duplicate label */
							fprintf(lst_fp,"** Line %u - Duplicate symbol: '%s'\n",line,label);
							++ecnt; }
						else {
							cflg=1;			/* indicate symbol table change */
							if(!(ptr = create_symbol(label, addr))) {
								xabort("symbol");
								fprintf(lst_fp,"** Line %u - Symbol table overflow\n", line);
								++ecnt;
								break; } } } }
				emsg = length = 0;
				if(itype >= 0x7E00) switch(itype) {
#include "UASM.002"
					case 0x7F01 :			/* EQU */
						if(!pflg) {
							set_symbol(ptr, get_value());
						err1st:
							if(emsg && !optf) {
								fprintf(lst_fp,"** Line %u - %s\n",line,etext[emsg]);
								++ecnt; } }
						break;
					case 0x7F02 :			/* ORG */
						addr=get_value();
						goto err1st;
					case 0x7F03 :			/* FCB */
						length=1;
						while(nxtelem()) ++length;
						break;
					case 0x7F04 :			/* FDB */
					case 0x7F05 :			/* RDB */
						length=2;
						while(nxtelem()) length += 2;
						break;
					case 0x7F06 :			/* FCC  */
					case 0x7F07 :			/* FCCH */
					case 0x7F08 :			/* FCCZ */
						chr=operand[0];
						for(i = 1; operand[i] && (operand[i] != chr); ++i)
							++length;
						if(itype == 0x7F08) ++length;
						break;
					case 0x7F09 :			/* RMB */
						addr = addr + get_value();
						goto err1st;
					case 0x7F0A :			/* END */
						goto end1;
					case 0x7F00+205 :		/* LINE */
						line = get_value(); }
				else
					calc_instruction(); }
			else
				length = 0;
			addr += length;
			++line; /* += (itype < (0x7F00+200)); */ }
end1:		++pflg;
		rewind(asm_fp); }
	while((optf)&&(cflg)&&(!ecnt));

/* second pass - generate object code */
	if(!quietf)
		fprintf(stderr,"Second pass... ");
	addr = emsg = 0; line = pcount = 1; lcount = 9999;
	while(readline()) {
		daddr = addr;
		if(optr) {
			itype = lookup_instruction(instruction);
			length = temp = 0;
			if(itype >= 0x7E00) switch(itype) {
#include "UASM.003"
				case 0x7F00 :			/* Bad instruction */
					reperr(ERRbadins);
					break;
				case 0x7F01 :			/* EQU */
					daddr=get_value();	/* generate errors if any */
					break;
				case 0x7F02 :			/* ORG */
					if(ocnt) wrrec();
					daddr=addr=get_value();
					break;
				case 0x7F03 :			/* FCB */
					do {
						instruction[length++] = get_value(); }
					while(*optr++ == ',');
					break;
				case 0x7F04 :			/* FDB */
				case 0x7F05 :			/* RDB */
					do {
						temp = get_value();
						if(itype == 0x7F04) {
							instruction[length++] = temp >> 8;
							instruction[length++] = temp; }
						else {
							instruction[length++] = temp;
							instruction[length++] = temp >> 8; } }
					while(*optr++ == ',');
					break;
				case 0x7F06 :			/* FCC  */
				case 0x7F07 :			/* FCCH */
				case 0x7F08 :			/* FCCZ */
					chr=operand[0];
					for(i = 1; operand[i] && (operand[i] != chr); ++i)
						instruction[length++] = operand[i];
					if(!operand[i])	reperr(ERRstring);
					if(itype == 0x7F07)
						instruction[length-1] |= 0x80;
					else if(itype == 0x7F08)
						instruction[length++] = 0;
					break;
				case 0x7F09 :			/* RMB statement */
					if(ocnt) wrrec();
					addr += get_value();
					break;
				case 0x7F0A :			/* END statement */
					goto end2;
				case 0x7F00+200 :		/* PAGE statement */
					lcount=9999;
					break;
				case 0x7F00+201 :		/* TITLE directive */
					strncpy(title, operand, sizeof(title)-1);
					break;
				case 0x7F00+202 :		/* SPACE directive */
					fprintf(lst_fp,"\n");
					++lcount;
					break;
				case 0x7F00+203 :		/* LIST directive */
					if(nlist) --nlist;
					break;
				case 0x7F00+204 :		/* NOLIST directive */
					++nlist;
					break;
				case 0x7F00+205 :		/* LINE directive */
					line = get_value(); }
			else
				calc_instruction(); }
		else
			length=itype=0;

/* generate listing */
	if(((itype < (0x7F00+200)) && fulf && !nlist) || emsg) {
		if(++lcount >= pagel)
			write_title();
		fprintf(lst_fp,"%04x ",daddr);
			for(i=0; i < 6; ++i) {
				if(i < length)
					fprintf(lst_fp," %02x",instruction[i] & 255);
				else
					fprintf(lst_fp,"   "); }
			fprintf(lst_fp," %c%5u  %s\n",(length <= 6) ? ' ' : '+',line,buffer); }
		if(emsg) {
			fprintf(lst_fp,"  ** ERROR ** - %u - %s\n", emsg, etext[emsg]);
			++ecnt; ++lcount;
			emsg=0; }
		++line; /* += ((itype < (0x7F00)+200)); */
/* write code to output record */
	for(i=0; i<length; ++i) {
		if(!ocnt) {			/* first byte of record */
			outrec[ocnt++]=addr>>8;
			outrec[ocnt++]=addr; }
		++addr;
		outrec[ocnt++]=instruction[i];
		if(ocnt>33) wrrec(); } }	/* record is full, write it */

end2:
	if(ocnt) wrrec();
	if(intel) fprintf(hex_fp,":00000001FF\n");
	else fprintf(hex_fp,"S9030000FC\n");

	if(ecnt) fprintf(lst_fp,"\n %u error(s) occurred in this assembly.\n",ecnt);
	if(!quietf) fprintf(stderr,"%u error(s).\n",ecnt);

/* display the symbol table */
	if(symf) {
		write_title();
		dump_symbols(); }

	fclose(asm_fp);
	fclose(hex_fp);
	fclose(lst_fp);

	return ecnt ? -2 : 0;
}

/*************************************************/
/* read a line from input file, and break it up. */
/*************************************************/
int readline()
{
	register int i;
	register char *ptr;

	if(!fgets(ptr = buffer, LINE_SIZE, asm_fp))
		return 0;

	i = 0;
	while(issymbol(*ptr)) {
		label[i] = chupper(*ptr++);
		if(i < SYMB_SIZE)
			++i; }
	label[i]=0;
	if(*ptr == ':')
		++ptr;
	while(isspace(*ptr))
		++ptr;
	if(((*ptr != '*') && (*ptr != ';')) || i) {
		i = 0;
		while(*ptr && !isspace(*ptr))
			instruction[i++] = toupper(*ptr++);
		instruction[i]=0;
		while(isspace(*ptr))
			++ptr;
		i = 0;
		while(*ptr && (i < (sizeof(operand)-1)))
			operand[i++] = *ptr++;
		(optr = operand)[i] = 0;
		return 1; }

	optr=0;
	return 1;
}

/* report an error */
reperr(errn)
	register int errn;
{	if(!emsg) emsg=errn;
}

/*******************************/
/* write record to output file */
/*******************************/
wrrec()
{	register unsigned i, chk, chr;

	xhex(ocnt-2);
	if(intel) {					/* intel hex format */
		chk = outrec[0] + outrec[1] + ocnt - 2;
		fprintf(hex_fp,":%02x%02x%02x00", ocnt-2, outrec[0] & 255, outrec[1] & 255);
		for(i=2; i<ocnt; ++i) {
			fprintf(hex_fp,"%02x", chr = outrec[i] & 255);
			chk += chr; }
		fprintf(hex_fp,"%02x\n", 255 & (0-chk)); }
	else {						/* motorola hex format */
		chk = ocnt + 1;
		fprintf(hex_fp,"S1%02x", ocnt + 1);
		for(i=0; i<ocnt; ++i) {
			fprintf(hex_fp,"%02x", chr = outrec[i] & 255);
			chk += chr; }
		fprintf(hex_fp,"%02x\n", 255 & ~chk); }
	ocnt = 0;
}

/*
 * Write out title
 */
write_title()
{
	int w;
	char *ptr;

	if(pcount > 1)
		putc('\f',lst_fp);
	lcount = 1;
	fprintf(lst_fp,"DUNFIELD "#CPU_LONG#" ASSEMBLER: ");
	ptr = title;
	for(w = 35; w < pagew; ++w)
		putc(*ptr ? *ptr++ : ' ', lst_fp);
	fprintf(lst_fp,"PAGE: %u\n\n", pcount);
	++pcount;
}

/**************************************/
/* find next element in argument list */
/**************************************/
int	nxtelem() {
	register char chr;
	while((chr = *optr) && (chr != ' ')&&(chr != 9)) {
		++optr;
		if(chr == '\'') {
			while((chr = *optr) && (chr != '\'')) ++optr;
			++optr; }
		if(chr==',') return(1); }
	return(0);
}

/*
 * Too many optimization passes - report failure
 */
optfail()
{
	fprintf(lst_fp,"** Line %u - Unable to resolve: %s\n", line, label);
	++ecnt;
}


/*
 * Lookup instruction in table
 */
int lookup_instruction(inst)
	char *inst;
{
	char *nptr;
	unsigned char *ptr;

	nptr = name_table;
	do {
		ptr = inst;
		while(*ptr == *nptr) {
			++ptr;
			++nptr; }
		if((*nptr < 0) && !*ptr) {
			ptr = nptr + 1;
			return ((*nptr & 0x7f) << 8) + *ptr; }
		while(*nptr > 0)
			++nptr; }
	while(*(nptr += 2));
	return *inst ? 0x7F00 : (0x7F00+199);
}

/*
 * Evaluate an expression
 */
get_value()
{
	unsigned v;

	v = get_simple_value();	
	for(;;) switch(*optr++) {
		case '+' :	v += get_simple_value();	continue;
		case '-' :	v -= get_simple_value();	continue;
		case '*' :	v *= get_simple_value();	continue;
		case '/' :	v /= get_simple_value();	continue;
		case '\\':	v %= get_simple_value();	continue;
		case '&' :	v &= get_simple_value();	continue;
		case '|' :	v |= get_simple_value();	continue;
		case '^' :	v ^= get_simple_value();	continue;
		case '>' :	v >>=get_simple_value();	continue;
		case '<' :	v <<=get_simple_value();	continue;
		case ')' :
		case ' ' :
		case '\t':
		case ',' :
		case 0   :
			--optr;
			return v;
		default: --optr;
			reperr(ERRexpre);
			return 0; }
}

/*
 * Evaluate a simple value
 */
get_simple_value()
{
	int c;
	unsigned v, i, b;
	char symname[50], *ptr;

	v=b=i=0;
	switch(c = *optr++) {
		case '$' :	b = 16;	goto getnum;
		case '@' :	b = 8;	goto getnum;
		case '%' :	b = 2;	goto getnum; }
	if(isdigit(c)) {
		symname[i++] = c;
	getnum:
		while(isalnum(c = toupper(*optr))) {
			symname[i++] = (c < 'A') ? c : c - 7;
			++optr; }
		if((b == 16) && !i)
			v = addr;
		else {
			if(!b) {
				b = 10;
				switch(symname[i-1]) {
					case 'H'-7 :	b = 16;	goto deci;
					case 'T'-7 :
					case 'D'-7 :	b = 10;	goto deci;
					case 'Q'-7 :
					case 'O'-7 :	b = 8;	goto deci;
					case 'B'-7 :	b = 2;
					deci:	--i;	} }
			if(!i)
				reperr(ERRexpre);
			for(ptr = symname; i; --i) {
				if((c = *ptr++ - '0') >= b)
					reperr(ERRexpre);
				v = (v * b) + c; } } }
	else switch(c) {
		case '\'' :		/* Quoted character */
			while(*optr) {
				if((c = *optr++) == '\'')
					return v;
				v = (v << 8) | c; }
			reperr(ERRstring);
			break;
		case '=' :		/* Reverse high & low values */
			v = get_simple_value();
			return (v >> 8) | (v << 8);
		case '-' :		/* Unary negate */
			return 0 - get_simple_value();
		case '~' :		/* Unary complement */
			return ~get_simple_value();
		case '*' :		/* Program counter */
			return addr;
		case '(' :
			v = get_value();
			if(*optr++ != ')') {
				reperr(ERRbrack);
				--optr; }
			break;
		default:
			if(!issymbol(c = *--optr)) {
				reperr(ERRexpre);
				break; }
			do
				symname[v++] = chupper(c);
			while(issymbol(c = *++optr));
			symname[v] = 0;
			if(lookup_symbol(symname))
				return value;
			reperr(ERRsymbol); }

	return v;
}

/*
 * Return valid symbol names
 */
issymbol(c)
	char c;
{
	switch(c) {			/* Special allowed characters */
		case '_' :
		case '?' :
		case '!' :
		case '.' :
			return 1; }

	return isalnum(c);
}

/*
 * Evaluate the operand types
 */
eval_operands(sizeflags)
	char sizeflags;
{
	int c, i, op, o[4];
	char *ptr, iflag, siflag, svflag, buffer[50];
	o[0] = o[1] = o[2] = o[3] = op = 0;

	do {
		ptr = optr;
		i = 0;
		while((c = *ptr++) && (c != ',') && !isspace(c))
			buffer[i++] = chupper(c);
		buffer[i] = 0;
		for(i=0; keyword_table[i]; ++i)
			if(!strcmp(keyword_table[i], buffer)) {
				i += KEYWORD;
				optr = ptr - 1;
				goto keyword; }

		iflag = siflag = svflag = 0;
		switch(sizeflags & (VWORD | VBYTE)) {
			case VBYTE : svflag = 1;	break;
			case VWORD : svflag = 2;	}
		switch(sizeflags & (IWORD | IBYTE)) {
			case IBYTE : siflag = 1;	break;
			case IWORD : siflag = 2;	}

		if(*optr == ',') {			/* Assumed '0' 1st operand */
			value = 0;
#ifdef	NULLOP
			i = NONE;
			goto keyword;
#endif
		} else {
			if(*optr == '#') {		/* Immediate mode */
				iflag = -1;
				++optr; }
			switch(*optr) {
				case '<' :			/* Force 8-bit */
					siflag = svflag = 1;
					++optr;
					break;
				case '>' :			/* Force 16-bit */
					siflag = svflag = 2;
					++optr; }
			value = get_value(); }

		if((!siflag) && ((int)value < 128) && ((int)value > -129))
			siflag = 1;
		if((!svflag) && (value < 256))
			svflag = 1;

		if(iflag)
			i = (siflag == 1) ? IMM8 : IMM16;
		else
			i = (svflag == 1) ? VAL8 : VAL16;

	keyword:
		o[op] = i;
		ovalue[op] = value;
		if(++op > 4)
			printf("Too many operands"); }
	while(*optr++ == ',');

	if((i = *(optr-1)) && !isspace(i))
		reperr(ERRformat);

	operand0 = (o[0] << 4) | o[1];
	operand1 = (o[2] << 4) | o[3];
}

/*
 * Calculate the instruction & place it in the buffer
 */
calc_instruction()
{
	int i, j, k, l;
	char *ptr;

	ptr = operand_table + itype;
	i = *ptr++;
	length = 0;
	if(ptr[1] || ptr[2]) {
		eval_operands(i);
		while(*ptr & 0x80) {
			if((ptr[1] != operand0) || (ptr[2] != operand1)) {
				ptr += (*ptr & 0x0f) + 3;
				continue; }
			while(!(*ptr & 0x7F))
				ptr += 3;
			goto decode; }
		reperr(ERRinvadr);
		return; }
decode:
	j = ((*ptr >> 4)  & 0x07) + 3;
	i = 3;
	while(i < j)
		instruction[length++] = ptr[i++];
	j = (*ptr & 0x0f) + 3;
	while(i < j) {
		if((l = ptr[i++]) & 0x80) {
			if(l == -1) {
				instruction[length++] = ptr[i++];
				continue; }
			k = user[l & 0x03]; }
		else
			k = ovalue[l & 0x03];
		if(l & 0x20)
			k = (k << 8) | ((unsigned)k >> 8);
		if(l & 0x10) {
			k = k - (RELCOMP);
			if(((k < -128) || (k > 127)) && !(l & 0x40)) {
				reperr(ERRrange); } }
#ifdef	LENDIAN
		instruction[length++] = k;
		if(l & 0x40)
			instruction[length++] = k >> 8;
#else
		if(l & 0x40)
			instruction[length++] = k >> 8;
		instruction[length++] = k;
#endif
		}
}
