/*
 * Universal Assembler Compiler
 *
 * ?COPY.TXT 2000-2005 Dave Dunfield
 *  -- see COPY.TXT --.
 *
 * DDS Micro-C/PC compile command: cc UAC -pof m=s
 */
// #define	DEBUG
#define	ASM_OUT
#include <stdio.h>

#define	MAX_KEYWORD	117			// Maximum number of keywords
#define	MAX_RANGE	10			// Maximum number of keyword ranges
#define	MAX_FUNC	256			// Maxumum number of functions
#define	MAX_INST	255			// Maximum number of instruction names
#define	MAX_OPER	255			// Maximum number of operand classes
#define	MAX_CLASS	5000		// Maximum number of inst/operand classes
#define	INST_SIZE	15			// Size of instruction
#define	OPER_SIZE	50			// Size of operand
#define	LINE_SIZE	200			// Size of input line
#define	NAME_SIZE	25			// Size of CPU name
#define	SYMB_SIZE	10			// number of extra symbols
#define	POOL_SIZE	45000		// Size of literal pool
#define	COMMENT_COL	40			// Column to align comments

#define	KEYWORD		0x80		// Beginning of keywords
#define	VALUE		0xF5		// Expression value
#define	KEYRANGE	0xF6		// Beginning of keyword ranges

#define	O_WORD		0x80		// Operand is a word
#define	O_REVERSE	0x40		// Operand is reversed
#define	O_FUNCTION	0x20		// Operand calls user function
#define	O_VALUE		0x10		// Operand or's with user value
#define	O_PREVIOUS	0x08		// Operand or's with previous value
#define	O_OPERAND	0x07		// Operand field

static FILE
	*ifp,						// Input file pointer
	*ofp;						// Output file pointer

static unsigned
	I, O,						// Instruction and operand counts
	Iindex,						// Instruction index
	Oindex,						// Operand index
	Ocount,						// Index of operand counts
	opcode_seg,					// Segment for opcode storage
	class[MAX_CLASS],			// Inst/Operand classification
	range_top = 0,				// Top of range list
	keyword_top = 0,			// Top of keyword list
	instruction_top = 0,		// Top of instruction list
	operand_top = 0,			// Top of operand list
	class_top = 0,				// Top of classification table
	opcode_top = 0,				// Top of opcode storage
	symbol_top = 0,				// Top of allowed character symbols
	last_function = -1,			// Records last function accessed
	line_number = 0;			// Current line number

static unsigned char
	input_buffer[LINE_SIZE+1],	// Input line buffer
	instruction[INST_SIZE+1],	// Instruction name buffer
	operand[OPER_SIZE+1],		// Operand storage buffer
	op_flags[8],				// Operand used flags
	function_flags[MAX_FUNC],	// Function defined table
	range_low[MAX_RANGE],		// Low range value
	range_high[MAX_RANGE],		// High range value
	symbol_chars[SYMB_SIZE+1],	// Allowed symbol characters
	cpu_name[NAME_SIZE+1],		// Cpu name
	*asm_name,					// Assembler name
	function_flag = 0,			// Function has been started flag
	*input_ptr,					// General input pointer
	*keywords[MAX_KEYWORD],		// Keyword list
	*instructions[MAX_INST],	// Instruction list
	*operands[MAX_OPER],		// Operand list
	pool[POOL_SIZE],			// Literal storage pool
	*pptr = &pool;				// Pool free pointer

static unsigned char lead_in[] = { "\
/*\n\
 * Universal Assembler Module for: \x81\n\
 *\n\
 * Compile to a module with DDS Micro-C/PC: cc \x80 -fmo\n\
 * Then link with UASM module: LC -s \x80 uasm\n\
 */\n\
extern unsigned Address, Length;\n\
extern void error(unsigned e);\n\
unsigned Code_function(unsigned Fn, unsigned x) {\n" };

/*
 * Formatted error message
 */
static register error(unsigned args)
{
	char buffer[100], c;
	unsigned i, m;

	_format_(nargs() * 2 + &args, buffer);
	printf("[%u]: ", line_number);
	fputs(buffer, stdout);
	putc('\n', stdout);
	m = 0;
	for(i=0; c = input_buffer[i]; ++i) {
		putc((c == '\t') ? ' ' : c, stdout);
		if((input_buffer+i) == input_ptr)
			m = i; }
	putc('\n', stdout);
	i = 0;
	while(m) {
		putc(' ', stdout);
		--m; }
	printf("^\n");

	exit(-1);
}

/*
 * Test to see if next character in the input stream is a "symbol"
 */
static int tstsymbol()
{
	unsigned i;
	for(i=0; i < symbol_top; ++i)
		if(symbol_chars[i] == *input_ptr)
			return 1;

	return isalpha(*input_ptr);
}

/*
 * Test to see if next character in the input stream is a "space"
 */
static int tstspace()
{
	return (*input_ptr == ' ') || (*input_ptr == '\t');
}

/*
 * Skip ahead to next non-space in the input stream
 */
static char skip_blanks()
{
	while(tstspace())
		++input_ptr;
	return *input_ptr;
}

/*
 * Report an error if a character is not found
 */
static void expect(char c)
{
	if(*input_ptr != c)
		error("Expected: '%c'", c);
	++input_ptr;
}

/*
 * Get a number in a number base
 */
static unsigned get_number(unsigned base)
{
	unsigned value, c;
	char f;

	value = f = 0;
	for(;;) {
		if(isdigit(c = *input_ptr))		/* convert numeric digits */
			c -= '0';
		else if(c >= 'a')				/* convert lower case alphabetics */
			c -= ('a' - 10);
		else if(c >= 'A')				/* convert upper case alphabetics */
			c -= ('A' - 10);
		else							/* not a valid "digit" */
			break;
		if(c >= base)					/* outside of base */
			break;
		value = (value * base) + c;		/* include in total */
		f = -1;
		++input_ptr; }
	if(!f)
		error("Base %u number expected!", base);
	return value;
}

/*
 * Add a string to the global literal pool
 */
static void add_pool(unsigned char *p)
{
	#ifdef DEBUG
		printf("Add pool: ");
	#endif
	do {
		#ifdef DEBUG
			if(isprint(*p))
				putc(*p, stdout);
			else
				printf("\\x%02x", *p);
		#endif
		*pptr++ = *p; }
	while(*p++);
	#ifdef DEBUG
		putc('\n', stdout);
	#endif
}

/*
 * Encode the opcode entry
 * Entry is added to the global opcode table in 'opcode_seg'.
 */
static void encode_opcode()
{
	unsigned operand, value, function;
	unsigned total_length, initial_length, code_length, length_mark;
	unsigned char xbits, initial;

	length_mark = opcode_top;
	opcode_top += 2;

	total_length = initial_length = code_length = initial = 0;

	if(!skip_blanks())
		error("No object code");
	#ifdef DEBUG
		printf("Encode:'%s'=", input_ptr);
	#endif

	do {
		operand = value = function = xbits = 0;
	next_char:
		switch(*input_ptr++) {
		case '>' :					/* WORD output */
			xbits |= O_WORD;
			goto next_char;
		case '=' :					/* Reverse/High output */
			xbits |= O_REVERSE;
	next_init:
			initial = -1;
			goto next_char;
		case '+' :					/* OR into previous opcode */
			xbits |= O_PREVIOUS;
			goto next_init;
		case '[' :					/* Operand specifier */
			if(*input_ptr != ':') {
				operand = get_number(10);
				if(operand >= Ocount)
					error("Operand not in arguments");
				op_flags[operand] = 0;
				++operand; }
			if(*input_ptr == ':') {	/* Function specifier */
				++input_ptr;
				xbits |= O_FUNCTION;
				function = get_number(10);
				if(function >= MAX_FUNC)
					error("Bad function");
				if(!function_flags[function])
					error("Function not defined"); }
			expect(']');
			goto next_init; }
		--input_ptr;
		if(isxdigit(*input_ptr)) {	/* HEX value */
			xbits |= O_VALUE;
			value = get_number(16);
			goto next_char; }

		if((!tstspace()) && *input_ptr)
			error("Unknown opcode encoding");

		if(!(xbits & O_VALUE)) {
			if(!operand)
				if(!(xbits & O_FUNCTION))
					error("No operand value"); }

		if(!initial) {
			poke(opcode_seg, opcode_top++, value);
			++initial_length;
			++total_length;
			if(xbits & O_WORD) {
				poke(opcode_seg, opcode_top++, value >> 8);
				++initial_length;
				++total_length; }
			continue; }

	poke(opcode_seg, opcode_top++, xbits | operand);
	++total_length;
	if(!(xbits & O_PREVIOUS)) {
		++code_length;
		if(xbits & O_WORD)
			++code_length; }

	if(xbits & O_FUNCTION) {
		poke(opcode_seg, opcode_top++, function);
		++total_length; }
	if(xbits & O_VALUE) {
		poke(opcode_seg, opcode_top++, value);
		++total_length;
		if(xbits & O_WORD) {
			poke(opcode_seg, opcode_top++, value >> 8);
			++total_length; } }
	}
	while(skip_blanks());

	poke(opcode_seg, length_mark, total_length+2);
	poke(opcode_seg, length_mark+1, (initial_length << 4) | code_length);

	for(operand=0; operand < Ocount; ++operand) {
		if(op_flags[operand])
			printf("[%u]: Operand %u not referenced\n", line_number, operand); }

	#ifdef DEBUG
		while(length_mark < opcode_top)
			printf(" %02x", peek(opcode_seg, length_mark++));
		printf("\n");
	#endif
}

/*
 * Write the lead in sequence if this is the first time
 */
write_lead_in()
{
	unsigned char *ptr, c;

	if(last_function == -1) {
		if(!*cpu_name)
			error("CPU_name (~$xxxx) must be defined before ~. or ~n");
		ptr = lead_in;
		while(c = *ptr++) {
			if(c & 0x80) switch(c) {
				case 0x80 :	fputs(asm_name, ofp); continue;
				case 0x81 : fputs(cpu_name, ofp); continue; }
			putc(c, ofp); }
		last_function = -2; }
}

/*
 * Read a line from the input source file and process it
 * - Parse instruction and operands, and fill in tables
 */
static int process_line()
{
	unsigned i, j;
	unsigned char temp[50], c;

	I = O = Ocount = 0;
again:
	if(!fgets(input_ptr = input_buffer, LINE_SIZE, ifp))
		return 0;
	++line_number;
	switch(skip_blanks()) {
	case ';' :			/* Comment - Ignore line */
	case 0   :			/* Null line - ignore it */
		goto again;
	case '~' :			/* Special function */
		if(isdigit(c = *++input_ptr)) {
			if((i = get_number(10)) >= MAX_FUNC)
				error("Function must be 0-%u", MAX_FUNC-1);
			skip_blanks();
			#ifdef DEBUG
				printf("ALG[%u]: %s\n", i, input_ptr);
			#endif
			write_lead_in();
			if(!function_flag) {
				fputs("switch(Fn) {\n", ofp);
				function_flag = -1; }
			if(!function_flags[i])
				fprintf(ofp, "case %u:", i);
			else if(i != last_function)
				error("Function code must be contiguous.");
			fputs(input_ptr, ofp);
			putc('\n', ofp);
			function_flags[last_function = i] = -1;
			goto again; }
		switch(c) {
		case '.' :		/* Define C lead-in */
			if(function_flag)
				error("Cannot lead-in after first function.");
			write_lead_in();
			++input_ptr;
			skip_blanks();
			fputs(input_ptr, ofp);
			putc('\n', ofp);
			goto again;
		case '+' :		/* Define extra symbol characters */
			while(c = *++input_ptr) {
				switch(c) {
				case ' ' :
				case '\t':
					continue;
				case '!' :
				case '"' :
				case '#' :
				case ',' :
				case '.' :
				case '?' :
				case '[' :
				case ']' :
				case '_' :
				case '`' :
				case '{' :
				case '}' :
					symbol_chars[symbol_top++] = c;
					continue; }
				error("Invalid symbol character: '%c'", c); }
			goto again;
		case '=' :		/* Define a keyword */
			++input_ptr;
			skip_blanks();
			for(;;) {
				if(!tstsymbol())
					error("Bad keyword");
				keywords[keyword_top] = pptr;
				while(tstsymbol() || isdigit(*input_ptr))
					*pptr++ = toupper(*input_ptr++);
				*pptr++ = 0;
				#ifdef DEBUG
					printf("KEYWORD[%u] ='%s'\n", keyword_top, keywords[keyword_top]);
				#endif
				++keyword_top;
				if(!skip_blanks())
					goto again;
				if(*input_ptr == ';')
					goto again; }
		case '$' :		/* Define CPU names */
			++input_ptr;
			skip_blanks();
			i = j = 0;
			while(*input_ptr && !isspace(*input_ptr))
				cpu_name[i++] = *input_ptr++;
			cpu_name[i] = 0;
			goto again;
		case '%' :		/* Define line range */
			if(range_top >= MAX_RANGE)
				error("Too many keyword ranges");
			++input_ptr;
			i = get_number(10);
			skip_blanks();
			j = get_number(10);
			if(i > j)
				error("Range must be ~%low high");
			if(j >= keyword_top)
				error("Range exceeds keywords");
			range_low[range_top] = i | 0x80;
			range_high[range_top] = j | 0x80;
			#ifdef DEBUG
				printf("RANGE[%u] = %u-%u\n", range_top,
					range_low[range_top], range_high[range_top]);
			#endif
			++range_top;
			goto again; }
		error("Unrecognized '~' command"); }

	/* Parse instruction */
	if(*input_ptr == '|')
		error("No source instruction");
	while((!tstspace()) && *input_ptr)
		instruction[I++] = toupper(*input_ptr++);
	instruction[I] = 0;

	/* Lookup instruction in list */
	for(Iindex = 0; Iindex < instruction_top; ++Iindex)
		if(!strcmp(instructions[Iindex], instruction))
			goto found_i;
	/* Instruction not found - add to pool */
	instructions[instruction_top++] = pptr;
	add_pool(instruction);
found_i:

	/* Parse operand */
	if(!skip_blanks())
		error("No source operand");
	while((!tstspace()) && (*input_ptr != '|') && *input_ptr) {
		if(tstsymbol()) {		/* Keyword */
			i = 0;
			while(tstsymbol() || isdigit(*input_ptr))
				temp[i++] = toupper(*input_ptr++);
			temp[i] = 0;
			for(i=0; i < keyword_top; ++i)
				if(!strcmp(keywords[i], temp))
					goto found_k;
			error("Keyword not found: %s", temp);
		found_k:
			operand[O++] = i | 0x80;
			op_flags[Ocount++] = 0;
			continue; }
		if(isdigit(*input_ptr))
			error("Operand conflicts with expression");
		switch(c = *input_ptr++) {
		case '$' :		/* Argument value */
			operand[O++] = VALUE;
			op_flags[Ocount++] = -1;
			continue;
		case '%' :		/* Keyword range */
			i = *input_ptr - '0';
			if(i >= range_top)
				error("Undefined range");
			++input_ptr;
			operand[O++] = KEYRANGE + i;
			op_flags[Ocount++] = -1;
			continue;
		case '~' :
		case '=' :
		case '@' :
		case '*' :
		case '(' :
		case '\'' :
			error("Operand conflicts with expression");
			continue; }
		operand[O++] = c; }
	operand[O] = 0;

	/* Lookup operand in list */
	for(Oindex = 0; Oindex < operand_top; ++ Oindex)
		if(!strcmp(operands[Oindex], operand))
			goto found_o;
	/* Operand not found - add to pool */
	operands[operand_top++] = pptr;
	add_pool(operand);
found_o:

	/* Process output translations */
	skip_blanks();
	expect('|');

	j = (Iindex << 8) | Oindex;
	for(i=0; i < class_top; ++i)
		if(class[i] == j)
			error("Duplicate instruction/operand class");

	class[class_top++] = j;

	encode_opcode();

	return -1;
}

/*
 * Setup for a comment to be output - space over to aligned column
 * and output opening sequence.
 */
static void begin_comment(unsigned l, unsigned column)
{
	do
		putc(' ', ofp);
	while(++l < column);
	fputs("/* ", ofp);
}

/*
 * Output a string as a comment
 */
static void print_operand(char *s)
{
	unsigned char c;
	while(c = *s++) {
		if(c == VALUE) {
			putc('$', ofp);
			continue; }
		if(c >= KEYRANGE) {
			fprintf(ofp, "%%%u", c - KEYRANGE);
			continue; }
		if(c >= KEYWORD) {
			fputs(keywords[c - KEYWORD], ofp);
			continue; }
		putc(c, ofp); }
}

/*
 * Print a string to the output file
 */
static unsigned print_string(unsigned char *s)
{
	unsigned l;
	unsigned char c;

	l = 2;
	putc('"', ofp);
	if(s) {
		while(c = *s++) {
			++l;
			if(isprint(c)) {
				putc(c, ofp);
				continue; }
			if(c == '\n') {
				fputs("\n", ofp);
				++l;
				continue; }
			fprintf(ofp, "\\x%02x", c);
			l += 3; } }
	putc('"', ofp);
	return l;
}

static void dump_data()
{
	unsigned i, j, k, l, t, t1, t2;
	unsigned char *ptr;

	fputs("int Tstsymbol(char c) {\n", ofp);
	if(symbol_top) {
		fputs("switch(c) {\n", ofp);
		for(i=0; i < symbol_top; ++i)
			fprintf(ofp, "case '%c':\n", symbol_chars[i]);
		fputs("\treturn -1; }\n", ofp); }
	fputs("\treturn Tstalnum(c); }\n", ofp);

#ifdef ASM_OUT
	fputs("asm { /* Instruction/Operand/Opcode translate table */\n", ofp);
	fputs("_Class_table:\n PUBLIC _Class_table\n", ofp);
	for(i=t2=0; t2 < class_top; ++t2) {
		k = class[t2];
		fprintf(ofp, "; %u: %s ", t2, instructions[k >> 8]);
		print_operand(operands[k & 255]);
		putc('\n', ofp);
		j = fprintf(ofp, " DB %u,%u", k & 255, k >> 8);
		j += fprintf(ofp, ",%u", t1 = k = peek(opcode_seg, i++));
		t = i;
		l = k - 1;
		while(l--)
			j += fprintf(ofp, ",%u", peek(opcode_seg, t++));
		do
			putc(' ', ofp);
		while(++j < COMMENT_COL);
		putc(';', ofp);
		k = peek(opcode_seg, i++);
		l = k >> 4;
		while(l--)
			fprintf(ofp, "%02x ", peek(opcode_seg, i++));
		l = (t1-2) - (k >> 4);
		while(l--) {
			k = peek(opcode_seg, i++);
			if(k & O_PREVIOUS)
				putc('+', ofp);
			if(k & O_WORD)
				putc('>', ofp);
			if(k & O_REVERSE)
				putc('=', ofp);
			if(k & (O_OPERAND|O_FUNCTION)) {
				putc('[', ofp);
				if(k & O_OPERAND)
					fprintf(ofp, "%u", (k & O_OPERAND)-1);
				if(k & O_FUNCTION) {
					--l;
					fprintf(ofp, ":%u", peek(opcode_seg, i++)); }
				putc(']', ofp); }
			if(k & O_VALUE) {
				--l;
				if(k & O_WORD) {
					--l;
					t = peek(opcode_seg, i++);
					t |= peek(opcode_seg, i++) << 8;
					fprintf(ofp, "%04x", t); }
				else
					fprintf(ofp, "%02x", peek(opcode_seg, i++)); }
			putc(' ', ofp); }
		putc('\n', ofp); }
	fputs(";\n DB 255,255\n }\n", ofp);
#else
	fputs("unsigned Class_table[] = { /* Instruction/Operand classification */", ofp);
	for(i=0; i < class_top; ++i) {
		if(i)
			putc(',', ofp);
		fputs("\n ", ofp);
		j = fprintf(ofp, "%u", k = class[i]);
		begin_comment(j, COMMENT_COL);
		fprintf(ofp, "%u: ", i);
		fputs(instructions[k >> 8], ofp);
		putc(' ', ofp);
		print_operand(operands[k & 255]);
		fputs(" */", ofp); }
	fputs(" };\n", ofp);

	fputs("unsigned char Opcode_data[] = { /* Opcode encoding data */", ofp);
	i = t2 = 0;
	while(i < opcode_top) {
		if(i)
			putc(',', ofp);
		fputs("\n ", ofp);
		j = fprintf(ofp, "%u", t1 = k = peek(opcode_seg, i++));
		t = i;
		l = k - 1;
		while(l--)
			j += fprintf(ofp, ",%u", peek(opcode_seg, t++));
		begin_comment(j, COMMENT_COL);
		fprintf(ofp, "%u: ", t2++);
		k = peek(opcode_seg, i++);
		l = k >> 4;
		while(l--)
			fprintf(ofp, "%02x ", peek(opcode_seg, i++));
		l = (t1-2) - (k >> 4);
		while(l--) {
			k = peek(opcode_seg, i++);
			if(k & O_PREVIOUS)
				putc('+', ofp);
			if(k & O_WORD)
				putc('>', ofp);
			if(k & O_REVERSE)
				putc('=', ofp);
			if(k & O_OPERAND)
				fprintf(ofp, "[%u", (k & O_OPERAND)-1);
			if(k & O_FUNCTION) {
				--l;
				fprintf(ofp, ":%u", peek(opcode_seg, i++)); }
			if(k & O_OPERAND)
				putc(']', ofp);
			if(k & O_VALUE) {
				--l;
				if(k & O_WORD) {
					--l;
					t = peek(opcode_seg, i++);
					t |= peek(opcode_seg, i++) << 8;
					fprintf(ofp, "%04x", t); }
				else
					fprintf(ofp, "%02x", peek(opcode_seg, i++)); }
			putc(' ', ofp); }
		fputs("*/", ofp); }
	fputs(" };\n", ofp);
#endif

	fputs("char *Keywords[] = { /* Keyword names */\n", ofp);
	for(i=0; i < keyword_top; ++i) {
		putc(' ', ofp);
		j = print_string(keywords[i]);
		begin_comment(j, COMMENT_COL);
		fprintf(ofp, "%u */,\n", i); }
	fputs(" 0 };\n", ofp);

	if(range_top) {
		fputs("unsigned char Range_low[] = { /* Keyword range - LOW */\n", ofp);
		for(i=0; i < range_top; ++i) {
			if(i)
				putc(',', ofp);
			fprintf(ofp, " %u", range_low[i]); }
		fputs(" };\n", ofp);

		fputs("unsigned char Range_high[] = { /* Keyword range - HIGH */\n", ofp);
		for(i=0; i < range_top; ++i) {
			if(i)
				putc(',', ofp);
			fprintf(ofp, " %u", range_high[i]); }
		fputs(" };\n", ofp); }
	else
		fputs("unsigned char Range_low[1], Range_high[1];\n", ofp);

	fputs("char *Instructions[] = { /* Instruction names */\n", ofp);
	for(i=0; i < instruction_top; ++i) {
		putc(' ', ofp);
		j = print_string(instructions[i]);
		begin_comment(j, COMMENT_COL);
		fprintf(ofp, "%u */,\n", i); }
	fputs(" 0 };\n", ofp);

	fputs("char *Operands[] = { /* Operand strings */\n", ofp);
	for(i=0; i < operand_top; ++i) {
		putc(' ', ofp);
		j = print_string(ptr = operands[i]);
		begin_comment(j, COMMENT_COL);
		fprintf(ofp, "%u: ", i);
		print_operand(ptr);
		fputs(" */,\n", ofp); }
	fputs(" 0 };\n", ofp);

	fprintf(ofp, "char CPU_name[] = { \"%s\" };\n", cpu_name);
	fprintf(ofp, "char ASM_name[] = { \"%s\" };\n", asm_name);
}

main(int argc, char *argv[])
{
	fputs("\nUniversal Assembler Compiler - 2.0\n", stderr);

	if(!(opcode_seg = alloc_seg(4096)))
		error("Not enough memory");

	if(argc < 2)
		abort("\nUse: uac <filename> [output file]\n\n?COPY.TXT 2000-2005 Dave Dunfield\n -- see COPY.TXT --.\n");

	concat(input_buffer, asm_name = argv[1], ".UAC");
	ifp = fopen(input_buffer, "rvq");
	if(argc > 2)
		asm_name = argv[2];
	concat(input_buffer, asm_name, ".C");
	ofp = fopen(input_buffer, "wvq");

	input_ptr = asm_name;
	while(*input_ptr) {
		switch(*input_ptr++) {
			case ':' :
			case '\\' :
				asm_name = input_ptr; } }

	while(process_line());
	write_lead_in();
	if(function_flag)
		fputs("} ", ofp);
	fputs("}\n", ofp);

	dump_data();

	fclose(ofp);
	fclose(ifp);


	free_seg(opcode_seg);

	for(I=O=0; I < MAX_FUNC; ++I)
		if(function_flags[I])
			++O;

	printf("\n--- Statistics ----   Used Avail\n");
	printf("Instruction names  : %5u %5u\n", instruction_top, MAX_INST);
	printf("Operand types      : %5u %5u\n", operand_top, MAX_OPER);
	printf("Opcode classes     : %5u %5u\n", class_top, MAX_CLASS);
	printf("Keyword names      : %5u %5u\n", keyword_top, MAX_KEYWORD);
	printf("Keyword ranges     : %5u %5u\n", range_top, MAX_RANGE);
	printf("Functions          : %5u %5u\n", O, MAX_FUNC);
	printf("Literal memory     : %5u %5u\n", pptr-pool, POOL_SIZE);
	printf("Translation memory : %5u %5u\n", opcode_top, -1);
}
