/*
 * Universal Assembler Main module
 *
 * ?COPY.TXT 2000-2005 Dave Dunfield
 *  -- see COPY.TXT --.
 *
 * Compile Commands:
 *   uac CPU                    <= Compile CPU.UAC to CPU.C
 *   cc CPU -fom                <= Create CPU.OBJ
 *   cc UASM -fmop ENDIAN=1     <= Create little endian UASM.OBJ
 *   cc UASM -fmop ENDIAN=2     <= Create big endian UASM.OBJ
 *   lc -s CPU UASM             <= Create CPU.EXE
 */

// #define DEBUG 3
#include <stdio.h>
#include <debug.h>

#if (ENDIAN < 1) || (ENDIAN > 2)
	#message "ENDIAN"=ENDIAN
	#error Set ENDIAN=1 for "Little endian" and ENDIAN=2 for "Big endian"
#endif

#define	VC_MAX			7		/* Maximum number of variables */
#define	VL_MAX			64		/* Maximum length of a variable */
#define	SYMB_SIZE		24		/* Maximum size of a symbol */
#define	LINE_SIZE		100		/* Maximum size of input line */
#define	CODE_SIZE		32		/* Size of output_file code record */
#ifndef DEMO
	#define	SYMB_TABLE	45000	/* Symbol table size */
#else
	#define	SYMB_TABLE	1200	/* Symbol table size */
	#define	LINE_LIMIT	1000	/* Line limit for demo */
#endif

#define	KEYWORD			0x80	/* Beginning of keywords */
#define	VALUE			0xF5	/* Expression value */
#define	KEYRANGE		0xF6	/* Beginning of keyword ranges */

#define	O_WORD			0x80	/* Operand is a word */
#define	O_REVERSE		0x40	/* Operand is reversed */
#define	O_FUNCTION		0x20	/* Operand calls user function */
#define	O_VALUE			0x10	/* Operand or's with user value */
#define	O_PREVIOUS		0x08	/* Operand or's with previous value */
#define	O_OPERAND		0x07	/* Operand field */

/* Flags in upper four bits of symbol table prefix entry */
#define SYMSORT			0x80	/* Symbol has been sorted */
#define	SYMMASK			0x1F	/* Mask for symbol length */

/* Error messages */
#define	ERR_inst		1		/* Unknown instruction */
#define	ERR_symbol		2		/* Undefined symbol */
#define	ERR_arg			3		/* Invalid argument format */
#define	ERR_mode		4		/* Invalid addressing mode */
#define	ERR_expr		5		/* Invalid expression syntax */
#define	ERR_range		6		/* Out of range */
#define	ERR_string		7		/* Improperly delimited string */

static char *error_text[] = { "?",
	"Unknown instruction",
	"Undefined symbol",
	"Invalid argument format",
	"Invalid addressing mode",
	"Expression syntax error",
	"Out or range",
	"Improperly delimited string" };

unsigned
	Length,						/* Length of instruction */
	Address;					/* Contains current PC address */

static unsigned
	Value,						/* Value returned from expression */
	Iclass,						/* Instruction classification number */
	Oclass,						/* Operand classification number */
	error_message = 0,			/* Pending error message */
	error_count = 0,			/* Count of errors */
	line_number = 0,			/* Current line number */
	line_count = 0,				/* Count of lines on page */
	page_count = 0,				/* Count of pages */
	code_count = 0,				/* Output code byte count */
	var_count,					/* Count of expression variables */
	pagel = 59,					/* Printer page length */
	pagew = 80;					/* Printer page width */

static unsigned char
	buffer[LINE_SIZE+1],		/* Line input buffer */
	label[SYMB_SIZE+1],			/* Label holding buffer */
	instruction[81],			/* Instruction holding buffer */
	operand[65],				/* Operand holding buffer */
	opclass[50],				/* Operand classification buffer */
	var_list[VC_MAX][VL_MAX+1],	/* Expression variables buffer */
	code_record[CODE_SIZE+2],	/* Code output record */
	*input_ptr,					/* General input parsing pointer */
	symbol_table[SYMB_TABLE],	/* Symbol table */
	title[50],					/* Title text for listing */
	*symbol_top = &symbol_table,/* Top of symbol table */
	list_inhibit = 0,			/* Listing inhibit flag */
	casf = 0,					/* Case sensitive flag */
	fulf = 0,					/* Full listing flag */
	intel = 0,					/* Intel output flag */
	quietf = 0,					/* Quiet mode flag */
	symf = 0;					/* Symbol table output flag */

static FILE
	*asm_fp,					/* .ASM input file pointer */
	*hex_fp,					/* .HEX output file pointer */
	*lst_fp;					/* .LST output file pointer */

static unsigned char *Directives[] = {
	"EQU",						/* 0 */
	"ORG",						/* 1 */
	"FCB",						/* 2 */
	"DB",						/* 3 */
	"FDB",						/* 4 */
	"DW",						/* 5 */
	"RDB",						/* 6 */
	"DRW",						/* 7 */
	"RMB",						/* 8 */
	"DS",						/* 9 */
	"FCC",						/* 10 */
	"STR",						/* 11 */
	"FCCZ",						/* 12 */
	"STRZ",						/* 13 */
	"FCCH",						/* 14 */
	"STRH",						/* 15 */
	"END",						/* 16 */
	"PAGE",						/* 17 */
	"SPACE",					/* 18 */
	"TITLE",					/* 19 */
	"LIST",						/* 20 */
	"NOLIST",					/* 21 */
	"LINE",						/* 22 */
	0 };

extern unsigned
	IOB_size;

extern unsigned Code_function(unsigned Fn, unsigned x);
extern int Tstsymbol(char c);		/* Test for valid symbol name */
extern char *Keywords[];			/* Keyword names */
extern unsigned char Range_low[];	/* Keyword range - LOW */
extern unsigned char Range_high[];	/* Keyword range - HIGH */
extern char *Instructions[];		/* Instruction names */
extern char *Operands[];			/* Operand strings */
extern char CPU_name[];				/* CPU name */
extern char ASM_name[];				/* Assembler name */

#ifdef DEMO
	#include "c:\project\demonote.txt"
	demo_abort(char *reason)
	{
		fprintf(stderr, "Demo version aborted because of to many %ss!\n", reason);
		fputs(demo_text, stderr);
		exit(-1);
	}
#endif

/* --- Misc. functions --- */

/*
 * Report an error
 */
void error(unsigned e)
{
	if(!error_message)
		error_message = e;
	return 0;
}

/*
 * Write a title
 */
static void write_title()
{
	int w;

	if(page_count)
		putc('\f', lst_fp);
	line_count = 0; /* 1 */
	w = fprintf(lst_fp, "DUNFIELD %s ASSEMBLER: %s", CPU_name, title) + 10;
	while(w++ < pagew)
		putc(' ', lst_fp);
	fprintf(lst_fp, "PAGE: %u\n\n", ++page_count);
}

/* --- Source parsing functions --- */

int Tstalnum(char c)
{
	return	((c >= 'a') && (c <= 'z'))
		||	((c >= 'A') && (c <= 'Z'))
		||	isdigit(c);
}

/*
 * Scan ahead next element in the argument list
 */
static unsigned next_element()
{
	char c;
	while(c = *input_ptr) {
		++input_ptr;
		if(c == '\'') {
			while((c = *input_ptr) && (c != '\''))
				++input_ptr;
			++input_ptr; }
		if(c == ',')
			return 1; }
	return 0;
}

/*
 * Convert character to upper case if NOT case sensitive
 */
static int chupper(char c)
{
	return casf ? c : ((c >= 'a') && (c <= 'z')) ? c - ('a'-'A') : c;
}

/*
 * Open a file with appropriate extension, and report any error
 */
static FILE *open_file(char *filename, char *extension, char *options)
{
	char buffer[100], *ptr, *dot;
	FILE *fp;

	dot = 0;

	for(ptr = buffer; *ptr = *filename; ++ptr) {	/* Copy filename */
		if(*filename == '.')
			dot = filename;
		if(*filename == '\\')
			dot = 0;
		++filename; }

	if(!dot) {					/* No extension supplied */
		*ptr++ = '.';
		do
			*ptr++ = *extension;
		while(*extension++); }
	else
		*dot = 0;

	if(!(fp = fopen(buffer, options))) {
		fprintf(stderr,"Unable to access: '%s'\n", buffer);
		exit(-1); }

	return fp;
}

/*
 * Read a line from the input file and break it up
 */
static int read_line()
{
	int i;
	unsigned char omode, c, *optr;
#ifdef DEMO
	static unsigned line_count = 0;
	if(++line_count >= LINE_LIMIT)
		demo_abort("line");
	if(!fgets(input_ptr = buffer, LINE_SIZE, asm_fp)) {
		line_count = -1;
		return 0; }
#else
	if(!fgets(input_ptr = buffer, LINE_SIZE, asm_fp))
		return 0;
#endif

	Debug2(("LINE: %s\n", buffer))

	++line_number;

	i = 0;
	while(Tstsymbol(*input_ptr)) {
		label[i] = chupper(*input_ptr++);
		if(i < SYMB_SIZE)
			++i; }
	label[i] = 0;
	if(*input_ptr == ':')
		++input_ptr;

	while(isspace(*input_ptr))
		++input_ptr;

	if((*input_ptr != '*') && (*input_ptr != ';') && *input_ptr) {
		i = omode = 0;
		while(*input_ptr && !isspace(*input_ptr))
			instruction[i++] = toupper(*input_ptr++);
		instruction[i] = 0;
		while(isspace(*input_ptr))
			++input_ptr;
		optr = operand;
		while(c = *input_ptr++) {
			if(omode) {			/* String input mode */
				if((*optr++ = c) == omode)
					omode = 0;
				continue; }
			switch(c) {
			case ';' :				/* Source comment */
				*optr = 0;
				input_ptr = operand;
				return -1;
			case '\'' :				/* Single quote */
			case '\"' :				/* Double quote */
				omode = c;
			default:				/* Unrecognized character */
				*optr++ = c;
			case ' ' :				/* Space character */
			case '\t' :				/* TAB character */
				continue; } }
		*optr = 0;
		if(omode)
			error(ERR_string);
		input_ptr = operand;
		Debug1(("L: '%s' I: '%s' O: '%s'\n", label, instruction, operand))
		return -1; }

	Debug1(("L: '%s' - No inst/oper\n", label))

	input_ptr = 0;

	return -1;
}

/* --- Symbol table management functions --- */

/*
 * Create a symbol table entry
 */
static unsigned char *create_symbol(char *s, unsigned v)
{
	unsigned char *ptr, *ptr1;

	ptr = ptr1 = symbol_top;
	while(*s)
		*++ptr = *s++;
	*ptr1 = ptr - ptr1;
	*++ptr = v >> 8;
	*++ptr = v;
	if(ptr < (symbol_table+sizeof(symbol_table))) {
		symbol_top = ptr + 1;
		return ptr1; }
#ifdef DEMO
	demo_abort("symbol");
#endif
	return 0;
}

/*
 * Lookup a symbol in the symbol table
 */
static unsigned char *lookup_symbol(unsigned char *s)
{
	unsigned l;
	unsigned char *ptr, *ptr1, *ptr2;

	Debug2(("Lookup symbol '%s' -", s))
	ptr = symbol_table;
again:
	if((ptr2 = ptr) >= symbol_top) {
		Debug2(("Not found!\n"))
		Value = 0x8888;
		return 0; }
	ptr1 = s;
	l = *ptr++ & SYMMASK;
	while(l--) {
		if(*ptr1++ != *ptr++) {
			ptr += l + 2;
			goto again; } }
	if(*ptr1) {
		ptr += 2;
		goto again; }
	Value = *ptr++ << 8;
	Value |= *ptr;
	Debug2(("value=%04x\n", Value))
	return ptr2;
}

/*
 * Set a value for a symbol table entry
 */
static void set_symbol(unsigned char *ptr, unsigned v)
{
	ptr += (*ptr & SYMMASK);
	*++ptr = v >> 8;
	*++ptr = v;
}

/*
 * Compare two symbol table entries
 */
static int compare_symbol(unsigned char *s1, unsigned char *s2)
{
	int l, l1, l2;
	l1 = *s1++ & SYMMASK;
	l2 = *s2++ & SYMMASK;
	l = (l1 < l2) ? l1 : l2;
	do {
		if(*s1 > *s2)
			return 1;
		if(*s1 < *s2)
			return -1; }
	while(--l);
	if(l1 > l2)
		return 1;
	if(l1 < l2)
		return -1;
	return 0;
}

/*
 * Display the symbol table (sorted)
 */
static void dump_symbols()
{
	unsigned char *ptr, *hptr;
	unsigned l, h, w;

	fprintf(lst_fp, "SYMBOL TABLE:\n\n");
	w = 0;
	for(;;) {
		for(hptr = symbol_table; hptr < symbol_top; hptr += (*hptr & SYMMASK) + 3)
			if(!(*hptr & SYMSORT))
				goto found;
		putc('\n', lst_fp);
		return;
found:	for(ptr = (*hptr & SYMMASK) + hptr + 3; ptr < symbol_top; ptr += (*ptr & SYMMASK) + 3) {
			if(*ptr & SYMSORT)
				continue;
			if(compare_symbol(ptr, hptr) < 0)
				hptr = ptr; }
		*(ptr = hptr) |= SYMSORT;
		h = l = *ptr++ & SYMMASK;
		if((w + l + 7) >= pagew) {			/* Overrun page width */
			putc('\n', lst_fp);
			w = 0; }
		if(w) {								/* Not start of line - separate */
			fprintf(lst_fp, "   ");
			w += 3; }
		while(l--)
			putc(*ptr++, lst_fp);
		w += (l = (h > 8) ? 24 : 8) + 5;	/* Calculate extended length */
		while(h++ < l)
			putc(' ', lst_fp);
		l = *ptr++ << 8;
		l |= *ptr++;
		fprintf(lst_fp, "-%04x", l); }
}

/* --- Operand classification functions --- */

/*
 * Skip a single element during classification
 */
static void skip_expression();	/* needed prototype */
static void skip_element()
{
	char c;
top:
	switch(c = *input_ptr++) {
		case '-' :		/* Negated value */
		case '~' :		/* Complemented value */
		case '=' :		/* Swap high and low bytes */
			goto top;
		case '(' :		/* Nested expression */
			skip_expression();
			if(*input_ptr == ')')
				++input_ptr;
			return; }
	switch(c) {
		case '$' :		/* Hex prefix */
		case '@' :		/* Octal prefix */
		case '%' :		/* Binary prefix */
			goto skip_num;
		case '\'' :		/* Quoted value */
			while(((c = *input_ptr++) != '\'') && c);
		case '*' :		/* Current PC value */
			return; }
	if(isdigit(c)) {
	skip_num:
		while(isxdigit(*input_ptr))
			++input_ptr;
		switch(toupper(*input_ptr)) {
			case 'H' :	/* Hex suffix */
			case 'O' :	/* Octal suffix */
			case 'Q' :	/* Octal suffix */
			case 'T' :	/* Decimal suffix */
				++input_ptr; }
		return; }

	if(Tstsymbol(c)) {
		while(Tstsymbol(*input_ptr))
			++input_ptr; }
}

/*
 * Skip a single expression value during classification
 */
static void skip_expression()
{
do_expr:
	skip_element();
	switch(*input_ptr) {
		case '+' :
		case '-' :
		case '*' :
		case '/' :
		case '\\':
		case '&' :
		case '|' :
		case '^' :
		case '<' :
		case '>' :
			++input_ptr;
			goto do_expr; }
}

/*
 * Locate a keyword name
 */
static int lookup_keyword()
{
	char *ptr, *ptr1;
	unsigned i;

	ptr = input_ptr;
	ptr1 = Keywords[i = 0];
	do {
		while(*ptr1) {
			if(chupper(*input_ptr++) != *ptr1++)
				goto skip; }
		if(!Tstsymbol(*input_ptr))
			return i | 0x80;
	skip: input_ptr = ptr; }
	while(ptr1 = Keywords[++i]);
	return 0;
}

/*
 * Classify an operand 
 */
static void classify_operand()
{
	unsigned char c, *oper_ptr, *ptr, *ptr1;

	Debug2(("Classify: %s\n", input_ptr))

	oper_ptr = opclass;
	var_count = 0;

	while(*input_ptr) {
		if(Tstsymbol(*input_ptr)) {
			if(c = lookup_keyword()) {
				*var_list[var_count++] = *oper_ptr++ = c;
				continue; }
		do_expr:
			ptr = input_ptr;
			skip_expression();
			ptr1 = var_list[var_count++];
			while((ptr < input_ptr) && *ptr)
				*ptr1++ = *ptr++;
			*ptr1 = 0;
			*oper_ptr++ = VALUE;
			continue; }
		switch(*input_ptr) {
			case '-' :		/* Unary negate */
				if(input_ptr[1] == '-')
					break;
				ptr = input_ptr++;
				if(lookup_keyword()) {
					input_ptr = ptr;
					break; }
				input_ptr = ptr;
			case '~' :		/* Unary compliment */
			case '=' :		/* Unary swap */
			case '$' :		/* Hex prefix */
			case '@' :		/* Octal prefix */
			case '%' :		/* Binary prefix */
			case '*' :		/* Program counter */
			case '(' :		/* Sub expression */
			case '\'':		/* Quoted constant */
				goto do_expr; }
		*oper_ptr++ = *input_ptr++; }
	*oper_ptr = 0;
}

/*
 * Match a classified operand string
 */
static unsigned lookup_operand()
{
	unsigned i;
	unsigned char *ptr, *ptr1, c, c1;

	Debug2(("Lookup operand: "))
	i = 0;
nomatch:
	while(ptr = Operands[i++]) {
		ptr1 = opclass;
		while(c = *ptr1++) {
			if((c1 = *ptr++) == c)
				continue;
			if(c1 >= KEYRANGE) {
				c1 -= KEYRANGE;
				if((c >= Range_low[c1]) && (c <= Range_high[c1]))
					continue; }
			goto nomatch; }
		if(*ptr)
			goto nomatch;
		Debug2(("Oclass=%u", i))
		return i; }
	Debug2(("Not found!\n"))
	return 0;
}

/*
 * Lookup an instruction
 */
static unsigned lookup_instruction()
{
	unsigned char  *ptr;

	Debug2(("Lookup instruction: '%s' - ", instruction))

	/* First, look for CPU instruction */
	for(Iclass = 0; ptr = Instructions[Iclass]; ++Iclass)
		if(!strcmp(instruction, ptr)) {
			Debug2(("Iclass = %u\n", Iclass))
			return 1; }

	/* Next, look for assembler directive */
	for(Iclass = 0; ptr = Directives[Iclass]; ++Iclass)
		if(!strcmp(instruction, ptr)) {
			Iclass |= 0x100;
			Debug1(("Iclass = %u\n", Iclass))
			return 2; }

	Debug2(("Not found!\n"))
	Iclass = -1;
	return 0;
}

/* --- Expression evaluation functions --- */

/*
 * Evaluate an expression.
 */
static unsigned get_value();	/* Needed prototype */
static unsigned evaluate()
{
	char c;
	unsigned result;

	result=get_value();
	while((c = *input_ptr++) && (c != ',')) switch(c) {
		case '+' : result += get_value();	continue;
		case '-' : result -= get_value();	continue;
		case '*' : result *= get_value();	continue;
		case '/' : result /= get_value();	continue;
		case '\\': result %= get_value();	continue;
		case '&' : result &= get_value();	continue;
		case '|' : result |= get_value();	continue;
		case '^' : result ^= get_value();	continue;
		case '<' : result <<= get_value();	continue;
		case '>' : result >>= get_value();	continue;
		default: error(ERR_expr); }
	--input_ptr;

	return result;
}

/*
 * Get a single value (number or symbol)
 */
static unsigned get_value()
{
	unsigned i, b, v;
	char c, array[25], *ptr;

	switch(c = *input_ptr++) {
		case '-' :				/* Negated value */
			return 0 - get_value();
		case '~' :				/* Complemented value */
			return ~get_value();
		case '=' :				/* Swap high and low bytes */
			i=get_value();
			return (i<<8)+(i>>8);
		case '(' :				/* Nested expression */
			v = evaluate();
			if(*input_ptr = ')')
				++input_ptr;
			else
				error(ERR_expr);
			return v; }

	v=b=i=0;
	switch(c) {
		case '$' :	b = 16;	goto getnum;
		case '@' :	b = 8;	goto getnum;
		case '%' :	b = 2;	goto getnum; }
	if(isdigit(c)) {
		array[i++] = c;
	getnum:
		while(Tstalnum(c = toupper(*input_ptr))) {
			array[i++] = (c < 'A') ? c : c - 7;
			++input_ptr; }
		if((b == 16) && !i)
			v = Address;
		else {
			if(!b) {
				b = 10;
				switch(array[i-1]) {
					case 'H'-7 :	b = 16;	goto deci;
					case 'T'-7 :
					case 'D'-7 :	b = 10;	goto deci;
					case 'Q'-7 :
					case 'O'-7 :	b = 8;	goto deci;
					case 'B'-7 :	b = 2;
					deci:	--i;	} }
			if(!i)
				error(ERR_expr);
			for(ptr = array; i; --i) {
				if((c = *ptr++ - '0') >= b)
					error(ERR_expr);
				v = (v * b) + c; } } }
	else if(c == '\'') {				/* Single quote. */
		while(((c = *input_ptr++) != '\'') && c)
			v = (v << 8) +c;
		if(!c)
			error(ERR_string); }
	else if(c == '*') {					/* Program counter */
		v = Address; }
	else {								/* Must be a label */
		i = 0;
		--input_ptr;
		while(Tstsymbol(c = chupper(*input_ptr))) {
			++input_ptr;
			label[i]=c;
			if(i < SYMB_SIZE)
				++i; }
		label[i]=0;

		if(!lookup_symbol(label))
			error(ERR_symbol);
		v = Value; }

	return v;
}

/* --- Code generation functions --- */

/*
 * Lookup a code table entry (Table in 80x86 code segment)
 */
static unsigned locate_code_table(io) asm
{
		extrn	_Class_table:NEAR
		MOV		AX,4[BP]				; Get pattern to look for
		MOV		BX,OFFSET _Class_table	; Point to class table
		PUSH	DS						; Save data segment
		PUSH	CS						; Get code segment
		POP		DS						; DS = code segment
		XOR		CH,CH					; Zero high
		MOV		DX,0FFFFh				; End indicator
look1:	CMP		AX,[BX]					; Does this match?
		JE		look2					; Yes, we found it
		ADD		BX,2					; Advance to next
		MOV		CL,[BX]					; Get total length
		ADD		BX,CX					; Skip to next entry
		CMP		DX,[BX]					; At end?
		JNE		look1					; Keep looking
		XOR		BX,BX					; Zero result
look2:	MOV		AX,BX					; Get pointer
		POP		DS						; Restore DS
}

/*
 * Get a byte from the code table (80x86 code segment)
 */
static unsigned get_code_byte(addr) asm
{
	MOV	BX,4[BP]
	MOV	AL,CS:[BX]
	XOR	AH,AH
}

/*
 * Write HEX record to output file
 */
static void write_hex_record()
{
	unsigned i, c, checksum;

	if(intel) {					/* intel hex format */
		checksum = code_record[0] + code_record[1] + code_count - 2;
		fprintf(hex_fp, ":%02x%02x%02x00", code_count-2, code_record[0], code_record[1]);
		for(i=2; i < code_count; ++i) {
			fprintf(hex_fp, "%02x", c = code_record[i]);
			checksum += c; }
		fprintf(hex_fp,"%02x\n", (-checksum) & 255); }
	else {						/* motorola hex format */
		checksum = code_count + 1;
		fprintf(hex_fp, "S1%02x", code_count + 1);
		for(i=0; i < code_count; ++i) {
			fprintf(hex_fp, "%02x", c = code_record[i]);
			checksum += c; }
		fprintf(hex_fp,"%02x\n", (~checksum) & 255); }
	code_count = 0;
}

/* --- Main program --- */

main(int argc, char *argv[])
{
	unsigned a, i, j, v, daddr, tl, temp;
	unsigned char lb, o;
	char c, *ptr, *lfile, *cfile;

	if(argc < 2) {
#ifdef DEMO
		printf("DDS %s Cross Assembler v2.0 (Demo)\n\nUse: %s <filename> [-cfiqst c=file l=file p=length w=width]\n",
#else
		printf("DDS %s Cross Assembler v2.0\n\nUse: %s <filename> [-cfiqst c=file l=file p=length w=width]\n",
#endif
			CPU_name, ASM_name);
		printf("\n?COPY.TXT 1983-2005 Dave Dunfield\n -- see COPY.TXT --.\n");
		exit(-1); }

	/* Parse for command line options */
	lfile = cfile = argv[1];
	for(i=2; i < argc; ++i) {
		if(*(ptr = argv[i]) == '-') {
			while(*++ptr) switch(toupper(*ptr)) {
				case 'C' : casf = -1;			continue;
				case 'F' : fulf = -1;			continue;
				case 'I' : intel = -1;			continue;
				case 'Q' : quietf = -1;			continue;
				case 'S' : symf = -1;			continue;
				case 'T' : lfile = 0;			continue;
				default: goto badopt; }
			continue; }
		if(*++ptr == '=') switch(toupper(*(ptr++ - 1))) {
			case 'L' : lfile = ptr;				continue;
			case 'C' : cfile = ptr;				continue;
			case 'P' : pagel = atoi(ptr)-1;		continue;
			case 'W' : pagew = atoi(ptr);		continue; }
	badopt:
		fprintf(stderr, "Invalid option: %s\n", argv[i]);
		exit(-1); }

	IOB_size = 1024;
	asm_fp = open_file(argv[1], "ASM", "r");
	IOB_size = 512;
	hex_fp = open_file(cfile, "HEX", "w");
	lst_fp = lfile ? open_file(lfile, "LST", "w") : stdout;

	strncpy(title, argv[1], sizeof(title)-1);

	/*
	 * First pass - Build symbol table
	 */
	Debug(("Pass 1\n"))
	if(!quietf)
		fputs("First pass... ", stderr);

	Address = 0;
	while(read_line()) {
		Length = error_message = 0;
		if(*label) {			/* Label exists */
			if(lookup_symbol(label)) {
				fprintf(lst_fp, "** Line %u - Duplicate symbol: %s\n", line_number, label);
				++error_count; }
			else if(!(ptr = create_symbol(label, Address))) {
				fprintf(lst_fp, "** Line %u - Symbol table overflow\n", line_number);
				++error_count;
				break; } }
		if(!input_ptr)
			continue;
		lookup_instruction();
		if(Iclass < 0x100) {	/* Standard instruction */
			classify_operand();
			if(Oclass = lookup_operand()) {
				i = locate_code_table((Iclass << 8) | (Oclass-1));
				Debug2(("Code table=%04x\n", i))
				j = get_code_byte(i+3);
				Length = (j >> 4) + (j & 0x0F);
				Debug2(("Length=%u\n", Length)) } }
		else switch(Iclass) {
		case 0x100+0 :			/* EQU */
			set_symbol(ptr, evaluate());
			break;
		case 0x100+1 :			/* ORG */
			Address = evaluate();
			break;
		case 0x100+2 :			/* FCB */
		case 0x100+3 :			/* DB */
			Length=1;
			while(next_element())
				++Length;
			break;
		case 0x100+4 :			/* FDB */
		case 0x100+5 :			/* DW */
		case 0x100+6 :			/* RDB */
		case 0x100+7 :			/* DRW */
			Length = 2;
			while(next_element())
				Length += 2;
			break;
		case 0x100+8 :			/* RMB */
		case 0x100+9 :			/* DS */
			Address += evaluate();
			break;
		case 0x100+12 :			/* FCCZ */
		case 0x100+13 :			/* STRZ */
			++Length;
		case 0x100+10 :			/* FCC */
		case 0x100+11 :			/* STR */
		case 0x100+14 :			/* FCCH */
		case 0x100+15 :			/* STRH */
			c = *input_ptr++;
			while((*input_ptr != c) && *input_ptr) {
				++Length;
				++input_ptr; }
			break;
		case 0x100+16 :			/* END */
			goto end_pass1;
		case 0x100+22 :			/* LINE */
			line_number = evaluate(); }

		if(error_message) {
			fprintf(lst_fp, "** Line %u - %s\n", line_number, error_text[error_message]);
			error_message = 0; }
		Address += Length; }

end_pass1:
	rewind(asm_fp);

	/*
	 * Second pass - Generate object code
	 */
	Debug(("Pass 2\n"))
	if(!quietf)
		fputs("Second pass... ", stderr);

	Address = line_number = 0;
	line_count = 9999;
	while(read_line()) {
		daddr = Address;
		Length = 0;
		if(!input_ptr)
			goto do_list;
		if(!lookup_instruction()) {
			error(ERR_inst);
			goto do_list; }
		if(Iclass < 0x100) {	/* Standard instruction */
			classify_operand();
			if(!(Oclass = lookup_operand())) {
				error(ERR_arg);
				goto do_list; }
			if(!(a = locate_code_table((Iclass << 8) | (Oclass - 1)))) {
				error(ERR_mode);
				goto do_list; }
			Debug2(("Code table=%04x\n", a))
			++a;							/* Skip pre-amble */
			tl = get_code_byte(++a) - 2;	/* Get total length */
			lb = get_code_byte(++a);		/* Get encoded output length */
			i = lb >> 4;					/* Get initial output size */
			while(i--) {					/* Generate initial data */
				instruction[Length++] = get_code_byte(++a);
				--tl; }
			while(tl--) {
				o = get_code_byte(++a);
				v = 0;
				if(i = o & O_OPERAND) {		/* Fetch operand value */
					if(*(input_ptr = var_list[i-1]) & 0x80)
						v = *input_ptr & 0x7F;
					else
						v = evaluate(); }
				if(o & O_FUNCTION) {		/* Apply user function */
					v = Code_function(get_code_byte(++a), v);
					--tl; }
				if(o & O_VALUE)  {			/* Include code value */
					v |= get_code_byte(++a);
					--tl; }
				if(o & O_REVERSE)			/* Reverse output */
					v = (v >> 8) | (v << 8);
#if ENDIAN==2
				if(o & O_PREVIOUS) {		/* Apply to previous opcode */
					if(o & O_WORD)			/* Word output */
						instruction[Length-2] |= v >> 8;
					instruction[Length-1] |= v; }
				else {						/* New opcode byte */
					if(o & O_WORD)
						instruction[Length++] = v >> 8;
					instruction[Length++] = v; } }
#else
				if(o & O_PREVIOUS) {		/* Apply to previous opcode */
					if(o & O_WORD) {		/* Word output */
						instruction[Length-2] |= v;
						instruction[Length-1] |= v >> 8; }
					else
						instruction[Length-1] |= v; }
				else {						/* New opcode byte */
					instruction[Length++] = v;
					if(o & O_WORD)
						instruction[Length++] = v >> 8; } }
#endif
			goto do_list; }
		switch(Iclass) {					/* Directives */
		case 0x100+0 :		/* EQU */
			daddr = evaluate();
			break;
		case 0x100+1 :		/* ORG */
			if(code_count)
				write_hex_record();
			Address = daddr = evaluate();
			break;
		case 0x100+2 :		/* FCB */
		case 0x100+3 :		/* DB */
			do {
				instruction[Length++] = evaluate(); }
			while(*input_ptr++ == ',');
			break;
		case 0x100+4 :		/* FDB */
		case 0x100+5 :		/* DW */
			Iclass = 0;
		case 0x100+6 :		/* RDB */
		case 0x100+7 :		/* DRW */
			do {
				temp = evaluate();
#if ENDIAN==2
				if(Iclass) {
					instruction[Length++] = temp;
					instruction[Length++] = temp >> 8; }
				else {
					instruction[Length++] = temp >> 8;
					instruction[Length++] = temp; } }
#else
				if(Iclass) {
					instruction[Length++] = temp >> 8;
					instruction[Length++] = temp; }
				else {
					instruction[Length++] = temp;
					instruction[Length++] = temp >> 8; } }
#endif
			while(*input_ptr++ == ',');
			break;
		case 0x100+8 :		/* RMB */
		case 0x100+9 :		/* DS */
			if(code_count)
				write_hex_record();
			Address += evaluate();
			break;
		case 0x100+10 :		/* FCC */
		case 0x100+11 :		/* STR */
		case 0x100+12 :		/* FCCZ */
		case 0x100+13 :		/* STRZ */
		case 0x100+14 :		/* FCCH */
		case 0x100+15 :		/* STRH */
			c = *input_ptr++;
			while((*input_ptr != c) && *input_ptr)
				instruction[Length++] = *input_ptr++;
			if(!*input_ptr)
				error(ERR_string);
			switch(Iclass) {
				case 0x100+12 :	/* FCCZ */
				case 0x100+13 :	/* STRZ */
					instruction[Length++] = 0;
					break;
				case 0x100+14 :	/* FCCH */
				case 0x100+15 :	/* STRH */
					instruction[Length-1] |= 0x80; }
			break;
		case 0x100+16 :		/* END */
			goto end_pass2;
		case 0x100+17 :		/* PAGE */
			line_count = 9999;
			break;
		case 0x100+18 :		/* SPACE */
			putc('\n', lst_fp);
			++line_count;
			break;
		case 0x100+19 :		/* TITLE */
			strncpy(title, operand, sizeof(title)-1);
			break;
		case 0x100+20 :		/* LIST */
			if(list_inhibit)
				--list_inhibit;
			break;
		case 0x100+21 :		/* NOLIST */
			++list_inhibit;
			break;
		case 0x100+22 :		/* LINE */
			line_number = evaluate(); }
	do_list:
		if(((Iclass <= (0x100+16)) && fulf && !list_inhibit) || error_message) {
			if(++line_count >= pagel)
				write_title();
			fprintf(lst_fp, "%04x ", daddr);
			for(i=0; i < 6; ++i) {
				if(i < Length)
					fprintf(lst_fp, " %02x", instruction[i]);
				else
					fputs("   ", lst_fp); }
			fprintf(lst_fp, " %c%5u  %s\n", (Length <= 6) ? ' ' : '+', line_number, buffer);
			if(error_message) {
				fprintf(lst_fp, "  ** ERROR ** - %u - %s\n", error_message, error_text[error_message]);
				++error_count;
				++line_count;
				error_message = 0; } }
		for(i=0; i < Length; ++i) {
			if(!code_count) {
				code_record[code_count++] = Address >> 8;
				code_record[code_count++] = Address; }
			++Address;
			code_record[code_count++] = instruction[i];
			if(code_count  > (CODE_SIZE+1))
				write_hex_record(); } }

end_pass2:
	if(code_count)
		write_hex_record();
	fputs(intel ? ":00000001FF\n" : "S9030000FC\n", hex_fp);

	if(error_count)
		fprintf(lst_fp, "\n %u error(s) occurred in this assembly.\n", error_count);

	if(!quietf)
		fprintf(stderr, "%u error(s).\n", error_count);

	if(symf) {
		write_title();
		dump_symbols(); }

	fclose(lst_fp);
	fclose(hex_fp);
	fclose(asm_fp);
}
