/*
 * Convert XASM format assembler source to INTEL format
 *
 * ?COPY.TXT 1992-2005 Dave Dunfield
 * **See COPY.TXT**.
 */
#include <stdio.h>

char *indirs[] = { "R0]", "R1]", "DPTR]", "A+DPTR]", "A+PC]", 0 };

char hexchar = 'H', binchar = 'B', octchar = 'Q', comchar = ';',
	labchar = ':', c8051 = -1;
int tab_size = 8, comment = 32;

extern unsigned IOB_size;

char help[] = "\n\
Performs most of the work necessary to convert the Motorola\n\
style source code used by XASM assemblers into Intel style.\n\n\
Use: xasm2int <infile> <outfile> [opts]\n\n\
opts:	B=B	Suffix for BINARY constants *\n\
	H=H	Suffix for HEX    constants *\n\
	O=Q	Suffix for OCTAL  constants *\n\
	L=:	Suffix for labels *\n\
	K=;	Comment indicator *\n\
	C=32	Column to assume comments\n\
	T=8	Width of TABS in input file\n\
	-@	Inhibit 8051 [] to @ conversions\n\n\
Default values are shown. Items marked with '*' can be\n\
disabled by specifying a null operand.\n\n\
?COPY.TXT 1992-2005 Dave Dunfield\n**See COPY.TXT**.\n";

/*
 * Test for str1 beginning with str2 (case insensitive)
 */
begins(str1, str2)
	char *str1, *str2;
{
	while(*str2)
		if(toupper(*str1++) != *str2++)
			return 0;
	return 1;
}

/*
 * Test for valid HEX character
 */
ishex(c)
	int c;
{
	return (c >= '0' && c <= '9')
		|| (c >= 'a' && c <= 'f')
		|| (c >= 'A' && c <= 'F');
}

/*
 * Main program
 */
main(argc, argv)
	int argc;
	char *argv[];
{
	int c, i, h, f;
	char buffer[500], out[500], *ptr, *ptr1;
	FILE *fpi, *fpo;

	if(argc < 3)
		abort(help);

	for(i=3; i < argc; ++i) {
		ptr = argv[i];
		switch((toupper(*ptr++)<<8) + toupper(*ptr++)) {
			case 'C=' :	comment = atoi(ptr);	break;
			case 'T=' :	tab_size = atoi(ptr);	break;
			case 'H=' :	hexchar = *ptr;			break;
			case 'O=' :	octchar = *ptr;			break;
			case 'B=' :	binchar = *ptr;			break;
			case 'K=' :	comchar = *ptr;			break;
			case 'L=' : labchar = *ptr;			break;
			case '-@' :	c8051 = 0;				break;
			default:
				fputs("Invalid option: ", stderr);
				abort(argv[i]); } }

	IOB_size = 8192;
	fpi = fopen(argv[1], "rvq");
	fpo = fopen(argv[2], "wvq");

	while(fgets(ptr = buffer, sizeof(buffer)-1, fpi)) {
		if((*ptr == '*') || (*ptr == ';')) {
			if(comchar)
				*ptr = comchar;
			fputs(buffer, fpo);
			putc('\n', fpo);
			continue; }
		ptr1 = out;
		h = f = -1;
top:	while(*ptr) {
			++h;
			switch(c = *ptr++) {
				case '\'' :		/* Single quoted string */
				case '"' :		/* Double quoted string */
					*ptr1++ = c;
					do
						*ptr1++ = *ptr;
					while(*ptr && (++h, *ptr++ != c));
					continue;
				case '$' :		/* Hex constant */
					if(hexchar) {
						if(!isdigit(*ptr))
							*ptr1++ = '0';
						while(ishex(*ptr))
							++h, *ptr1++ = *ptr++;
						*ptr1++ = hexchar;
						continue; }
					break;
				case '%' :		/* Binary constant */
					if(binchar) {
						while((*ptr =='0') || (*ptr == '1'))
							++h, *ptr1++ = *ptr++;
						*ptr1++ = binchar;
						continue; }
					break;
				case '@' :		/* Octal constant */
					if(octchar) {
						while((*ptr >= '0') && (*ptr <= '7'))
							++h, *ptr1++ = *ptr++;
						*ptr1++ = octchar;
						continue; }
					break;
				case 9 :		/* Horizontal TAB */
				case ' ' :		/* Space */
					if(h && f && labchar)
						*ptr1++ = labchar;
					if((*ptr1++ = c) == 9) {
						while(++h % tab_size);
						--h; }
					f = 0;
					continue;
				case '[' :		/* Indirection */
					if(c8051) {
						for(i=0; indirs[i]; ++i)
							if(begins(ptr, indirs[i])) {
								*ptr1++ = '@';
								while(*ptr != ']')
									++h, *ptr1++ = *ptr++;
								++h, ++ptr;
								goto top; } } }

			if((h >= comment) && (h < 1000) && comchar) {
				h=1000;
				if(c != comchar)
					*ptr1++ = comchar; }
			*ptr1++ = c; }

		*ptr1 = 0;
		fputs(out, fpo);
		putc('\n', fpo); }

	fclose(fpo);
	fclose(fpi);
}
